(ns farbetter.pete
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [go-safe inspect sym-map])]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [go-safe inspect sym-map]])))


(def default-master-fn-interval-ms 5)
(def default-master-loop-interval-ms 100)

;;;;;;;;;;;;;;;;;;;; Protocol and Record Defs ;;;;;;;;;;;;;;;

(defprotocol IRepeater
  (add-fn [this fn-name fn interval-ms])
  (remove-fn [this fn-name])
  (start [this] "Restarts the calling of fns after stop has been called.")
  (stop [this] "Stops the calling of fns.")
  (-start-master-loop [this] "Internal use only.")
  (-start-repeat-loop [this fn-name] "Internal use only."))

(defrecord Repeater  [fn-info-atom running?-atom
                      master-fn-interval-ms master-loop-interval-ms]
  IRepeater
  (add-fn [this fn-name f interval-ms]
    (when (@fn-info-atom fn-name)
      (throw-far-error (str "Fn name `" fn-name "` already exists.")
                       :illegal-argument :fn-name-already-exists
                       (sym-map fn-name)))
    (let [loop-started?-atom (atom false)]
      (swap! fn-info-atom assoc fn-name
             (sym-map f interval-ms loop-started?-atom)))
    nil)

  (remove-fn [this fn-name]
    (swap! fn-info-atom dissoc fn-name)
    nil)

  (start [this]
    (when-not @running?-atom
      (reset! running?-atom true)
      (-start-master-loop this))
    nil)

  (stop [this]
    (debugf "Entering pete/stop. @running?-atom: %s" @running?-atom)
    (when @running?-atom
      (reset! running?-atom false)
      (doseq [[fn-name {:keys [loop-started?-atom]}] @fn-info-atom]
        (when loop-started?-atom
          (reset! loop-started?-atom false))))
    (debugf "Exiting pete/stop. @running?-atom: %s" @running?-atom)
    nil)

  (-start-master-loop [this]
    (go-safe
     (while @running?-atom
       (doseq [[fn-name {:keys [loop-started?-atom]}] @fn-info-atom]
         (when loop-started?-atom ;; Make sure it's not nil
           (when-not @loop-started?-atom
             (-start-repeat-loop this fn-name)
             (reset! loop-started?-atom true)))
         (ca/<! (ca/timeout master-fn-interval-ms)))
       (ca/<! (ca/timeout master-loop-interval-ms))))
    nil)

  (-start-repeat-loop [this fn-name]
    (debugf "Starting repeat loop for %s" fn-name)
    (go-safe
     (loop []
       (debugf "At top of repeat loop for %s. @running?-atom: %s"
               fn-name @running?-atom)
       (let [{:keys [interval-ms f]} (@fn-info-atom fn-name)]
         (when (and @running?-atom f)
           (f)
           (ca/<! (ca/timeout interval-ms))
           (recur)))))
    nil))

;;;;;;;;;;;;;;;;;;;; Constructor ;;;;;;;;;;;;;;;;;;;;

(s/defn make-repeater :- (s/protocol IRepeater)
  ([]
   (make-repeater default-master-fn-interval-ms
                  default-master-loop-interval-ms))
  ([master-fn-interval-ms :- s/Int
    master-loop-interval-ms :- s/Int]
   (let [fn-info-atom (atom {})
         running?-atom (atom false)
         params (sym-map fn-info-atom running?-atom master-fn-interval-ms
                         master-loop-interval-ms)
         repeater (map->Repeater params)]
     (start repeater)
     repeater)))
