(ns farbetter.pete
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [inspect sym-map])]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])))

(declare schedule-fn)

(def TimeMs s/Num)
(def FnKey s/Keyword)
(def FnInfo {:f (s/=> s/Any)
             :interval-ms TimeMs})
(def FnMap {FnKey FnInfo})
(def Schedule {TimeMs [FnKey]})
(def ScheduleEntry [(s/one TimeMs "time to run")
                    (s/one [FnKey] "fn-keys")])

;;;;;;;;;;;;;;;;;;;; Protocol and Record Defs ;;;;;;;;;;;;;;;

(defprotocol IRepeater
  (add-fn! [this k f interval-ms]
    "Add a fn to be repeated.
     Parameters:
     - this - A repeater instance.
     - k - A keyword to uniquely identify the fn.
     - f - The fn to be called.
     - interval-ms - The call interval in milliseconds.")
  (remove-fn! [this k]
    "Removes the fn referred to by `k`")
  (start [this] "Restarts the calling of fns after stop has been called.")
  (stop [this] "Stops the calling of fns."))

(defrecord Repeater  [active?-atom fn-map-atom schedule-atom]
  IRepeater
  (add-fn! [this k f interval-ms]
    (when (contains? @fn-map-atom k)
      (throw-far-error (str "Key `" k "` already exists.")
                       :illegal-argument :duplicate-fn-key (sym-map k)))
    (swap! fn-map-atom assoc k (sym-map f interval-ms))
    (swap! schedule-atom
           #(schedule-fn @fn-map-atom (u/get-current-time-ms) % k))
    nil)

  (remove-fn! [this k]
    (swap! fn-map-atom dissoc k)
    nil)

  (start [this]
    (reset! active?-atom true)
    nil)

  (stop [this]
    (reset! active?-atom false)
    nil))

;;;;;;;;;;;;;;;;;;;; Helper Fns ;;;;;;;;;;;;;;;;;;;;

(s/defn schedule-fn :- Schedule
  [fn-map :- FnMap
   old-run-time :- TimeMs
   schedule :- Schedule
   fn-key :- FnKey]
  (if-let [fn-info (fn-map fn-key)]
    (let [new-run-time (+ old-run-time (:interval-ms fn-info))]
      (update schedule new-run-time #(-> (or % [])
                                         (conj fn-key))))
    schedule))

(s/defn schedule-entry :- Schedule
  [fn-map :- FnMap
   schedule :- Schedule
   entry :- ScheduleEntry]
  (let [[old-run-time fn-keys] entry
        schedule (dissoc schedule old-run-time)]
    (reduce (partial schedule-fn fn-map old-run-time)
            schedule fn-keys)))

(s/defn update-schedule :- Schedule
  [schedule :- Schedule
   entries :- [ScheduleEntry]
   fn-map :- FnMap]
  (reduce (partial schedule-entry fn-map)
          schedule entries))

(s/defn run-fns :- (s/eq nil)
  [fn-map :- FnMap
   entries :- [ScheduleEntry]]
  (doseq [[run-time fn-keys] entries]
    (doseq [fn-key fn-keys]
      (when-let [fn-info (fn-map fn-key)]
        (u/go-sf
         ((:f fn-info)))))))

(s/defn start-main-loop :- (s/eq nil)
  [active?-atom :- (s/atom s/Bool)
   fn-map-atom :- (s/atom FnMap)
   schedule-atom :- (s/atom Schedule)]
  (u/go-sf
   (while @active?-atom
     (when-let [to-run-seq (subseq @schedule-atom
                                   <= (u/get-current-time-ms))]
       (run-fns @fn-map-atom to-run-seq)
       (swap! schedule-atom update-schedule to-run-seq @fn-map-atom))
     (ca/<! (ca/timeout 1))))
  nil)

;;;;;;;;;;;;;;;;;;;; Constructor ;;;;;;;;;;;;;;;;;;;;

(s/defn make-repeater :- (s/protocol IRepeater)
  []
  (let [active?-atom (atom true)
        fn-map-atom (atom {})
        schedule-atom (atom (sorted-map))
        repeater (->Repeater active?-atom fn-map-atom schedule-atom)]
    (start-main-loop active?-atom fn-map-atom schedule-atom)
    repeater))
