(ns farbetter.pete
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [go-safe inspect sym-map])]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [go-safe inspect sym-map]])))

(declare start-master-loop)

(def default-master-fn-interval-ms 5)
(def default-master-loop-interval-ms 100)

;;;;;;;;;;;;;;;;;;;; Protocol and Record Defs ;;;;;;;;;;;;;;;

(defprotocol IRepeater
  (add-fn [this fn-name fn interval-ms])
  (remove-fn [this fn-name])
  (start [this] "Restarts the calling of fns after stop has been called.")
  (stop [this] "Stops the calling of fns."))

(defrecord Repeater  [fn-info-atom running?-atom
                      master-fn-interval-ms master-loop-interval-ms]
  IRepeater
  (add-fn [this fn-name fn interval-ms]
    (when (@fn-info-atom fn-name)
      (throw-far-error (str "Fn name `" fn-name "` already exists.")
                       :illegal-argument :fn-name-already-exists
                       (sym-map fn-name)))
    (swap! fn-info-atom assoc fn-name (sym-map fn interval-ms)))
  (remove-fn [this fn-name]
    (swap! fn-info-atom dissoc fn-name))
  (start [this]
    (when-not @running?-atom
      (reset! running?-atom true)
      (start-master-loop this)))
  (stop [this]
    (reset! running?-atom false)))

;;;;;;;;;;;;;;;;;;;; Constructor ;;;;;;;;;;;;;;;;;;;;

(s/defn make-repeater :- (s/protocol IRepeater)
  ([]
   (make-repeater default-master-fn-interval-ms
                  default-master-loop-interval-ms))
  ([master-fn-interval-ms :- s/Int
    master-loop-interval-ms :- s/Int]
   (let [fn-info-atom (atom {})
         running?-atom (atom false)
         params (sym-map fn-info-atom running?-atom master-fn-interval-ms
                         master-loop-interval-ms)
         repeater (map->Repeater params)]
     (start repeater)
     repeater)))

;;;;;;;;;;;;;;;;;;;; Helper fns ;;;;;;;;;;;;;;;;;;;;

(defn start-repeat-loop [info-atom fn-name]
  (let [{:keys [interval-ms fn]} (@info-atom fn-name)]
    (go-safe
     (while (@info-atom fn-name)
       (fn)
       (ca/<! (ca/timeout interval-ms))))))

(defn start-master-loop [repeater]
  (go-safe
   (while @(:running?-atom repeater)
     (let [{:keys [fn-info-atom master-fn-interval-ms]} repeater
           {:keys [master-loop-interval-ms]} repeater]
       (doseq [[fn-name {:keys [loop-started]}] @fn-info-atom]
         (when-not loop-started
           (start-repeat-loop fn-info-atom fn-name)
           (swap! fn-info-atom assoc-in [fn-name :loop-started] true))
         (ca/<! (ca/timeout master-fn-interval-ms)))
       (ca/<! (ca/timeout master-loop-interval-ms))))))
