(ns farbetter.mu.msgs
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.roe :as roe]
   [farbetter.roe.schemas :as rs]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [go-safe inspect sym-map]
                         :cljs [byte-array long])]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [go-safe inspect sym-map]])))

(def gws-uri "http://zeus.farbetter.com/gateways")
(def gws-schema {:type :array
                 :items :string})

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; API Schemas ;;;;;;;;;;;;;;;;;;;;;;;;;

(def ServiceName s/Str)
(def MajorVersion s/Num)
(def FnName s/Str)
(def FnMap {FnName (s/=> s/Any s/Any)})
(def FnSchemas {(s/required-key :arg-schema) rs/AvroSchema
                (s/required-key :return-schema) rs/AvroSchema})
(def FnSchemaMap {FnName FnSchemas})
(def APISpec {(s/required-key :service-name) ServiceName
              (s/required-key :major-version) MajorVersion
              (s/required-key :fn-schemas) FnSchemaMap})

;;;;;;;;;;;;;;;;;;;; Avro schemas for messages ;;;;;;;;;;;;;;;;;;;;

(def uuid-schema
  {:name :uuid
   :type :record
   :doc "UUID"
   :fields [{:name :high
             :type :int
             :doc "Highest 32 bits of the UUID"
             :default 0}
            {:name :midh
             :type :int
             :doc "Second-highest 32 bits of the UUID"
             :default 0}
            {:name :midl
             :type :int
             :doc "Second-lowest 32 bits of the UUID"
             :default 0}
            {:name :low
             :type :int
             :doc "Lowest 32 bits of the UUID"
             :default 0}]})

(def uuid-default (roe/make-default-record uuid-schema))
(def fingerprint-schema :long)
(def fingerprint-default (u/long 0))
(def user-id-type uuid-schema)
(def user-id-default uuid-default)
(def request-id-type uuid-schema)
(def request-id-default uuid-default)
(def fn-name-type :string)
(def fn-name-default "no fn name")

(def fragment-schema
  {:name :fragment
   :type :record
   :doc "Small messages will be sent as a single fragment. Large messages
         will be broken into multiple fragments."
   :fields [{:name :msg-id
             :type [:null uuid-schema]
             :doc "Unique identifier for msg. Present in all fragments of
                   multi-fragment msgs. Null for single-fragment msgs."
             :default nil}
            {:name :num-fragments
             :type :int
             :doc "Number of fragments in this msg. Must be 1 or greater."
             :default 1}
            {:name :fragment-num
             :type :int
             :doc "Fragment number (zero-based)."
             :default 0}
            {:name :fingerprint
             :type fingerprint-schema
             :default fingerprint-default
             :doc "Avro schema fingerprint for message."}
            {:name :fragment-bytes
             :type :bytes
             :doc "Avro binary-encoded message data. Treated as opaque."
             :default (byte-array [])}]})

(def schema-rq-schema
  {:name :schema-rq
   :type :record
   :doc "Message to be sent when a schema's fingerprint is unknown."
   :fields [{:name :fingerprint
             :type fingerprint-schema
             :doc "Fingerprint of the schema"
             :default fingerprint-default}]})

(def schema-rs-schema
  {:name :schema-rs
   :type :record
   :doc (str "Maps a schema's fingerprint to a json "
             "represenation of that schema.")
   :fields [{:name :fingerprint
             :type fingerprint-schema
             :doc "Fingerprint of the schema."
             :default fingerprint-default}
            {:name :json-schema
             :type [:null :string]
             :doc "JSON representation of the schema. Null if not found."
             :default nil}]})

(def service-api-version-schema
  {:name :service-api-version
   :type :record
   :doc "Major API version of a service."
   :fields [{:name :service-name
             :type :string
             :doc "Name of the service. May contain dots."
             :default "no-name"}
            {:name :major-version
             :type :int
             :default 0}]})

(def service-instance-version-schema
  {:name :service-instance-version
   :type :record
   :doc "Full version of a service"
   :fields [{:name :service-name
             :type :string
             :doc "Name of the service. May contain dots."
             :default "no-name"}
            {:name :major-version
             :type :int
             :default 0}
            {:name :minor-version
             :type :int
             :default 0}
            {:name :micro-version
             :type :int
             :default 0}]})

(def client-login-rq-schema
  {:name :client-login-rq
   :type :record
   :fields [{:name :email
             :type :string
             :doc "The user's email address"
             :default ""}
            {:name :password
             :type :string
             :doc "The user's password"
             :default ""}]})

(def client-login-rs-schema
  {:name :client-login-rs
   :type :record
   :fields [{:name :was-successful
             :type :boolean
             :default false}
            {:name :user-id
             :type [:null uuid-schema]
             :default nil}]})

(def service-instance-login-rq-schema
  {:name :service-instance-login-rq
   :type :record
   :fields [{:name :access-key
             :type :string
             :default "no key"}
            {:name :service-instance-version
             :type service-instance-version-schema
             :default (roe/make-default-record
                       service-instance-version-schema)}]})

(def service-instance-login-rs-schema
  {:name :service-instance-login-rs
   :type :record
   :fields [{:name :was-successful
             :type :boolean
             :default false}]})

(def rpc-rs-success-schema
  {:name :rpc-rs-success
   :type :record
   :fields [{:name :request-id
             :type request-id-type
             :default request-id-default}
            {:name :return-schema-fp
             :type fingerprint-schema
             :doc "Fingerprint of the return value schema"
             :default fingerprint-default}
            {:name :encoded-return-value
             :type :bytes
             :doc (str "Return value of the fn. Avro binary-encoded data. "
                       "Treated as opaque.")
             :default (byte-array [])}]})

(def rpc-failure-reason-schema
  {:name :rpc-failure-reason
   :type :record
   :fields [{:name :type
             :type :string
             :default "no type"}
            {:name :subtype
             :type :string
             :default "no subtype"}
            {:name :encoded-error-map
             :type [:null :string]
             :default nil}
            {:name :exception-msg
             :type :string
             :default "no msg"}
            {:name :stacktrace
             :type :string
             :default "no stacktrace"}
            {:name :request-id-str
             :type :string
             :default "no rq id"}
            {:name :fn-name
             :type :string
             :default "no fn name"}]})

(def rpc-rs-failure-schema
  {:name :rpc-rs-failure
   :type :record
   :fields [{:name :request-id
             :type request-id-type
             :default request-id-default}
            {:name :reason
             :type rpc-failure-reason-schema
             :default (roe/make-default-record
                       rpc-failure-reason-schema)}]})

(def rpc-rq-schema
  {:name :rpc-rq
   :type :record
   :fields [{:name :service-api-version
             :type service-api-version-schema
             :default (roe/make-default-record service-api-version-schema)}
            {:name :user-id
             :type user-id-type
             :default user-id-default}
            {:name :request-id
             :type request-id-type
             :default request-id-default}
            {:name :fn-name
             :type fn-name-type
             :doc "The name of the function to be called. May contain dots."
             :default fn-name-default}
            {:name :arg-schema-fp
             :type fingerprint-schema
             :doc "Fingerprint of the arg schema"
             :default fingerprint-default}
            {:name :encoded-arg
             :type :bytes
             :doc (str "Avro binary-encoded function argument data. "
                       "Treated as opaque.")
             :default (byte-array [])}
            {:name :timeout-ms
             :type :int
             :doc (str "Timeout in milliseconds. RPC will canceled after "
                       "this much time has passed.")
             :default (* 1000 60)}]})

(def traffic-policy-rule-schema
  {:name :traffic-policy-rule
   :type :record
   :fields [{:name :users
             :type {:type :array
                    :items user-id-type}
             :default []}
            {:name :weight
             :type :int
             :default 1}]})

(def traffic-policy-schema
  {:name :traffic-policy
   :type :record
   :fields [{:name :service-instance-version
             :type service-instance-version-schema
             :default (roe/make-default-record
                       service-instance-version-schema)}
            {:name :rules
             :type {:type :array
                    :items traffic-policy-rule-schema}
             :default [(roe/make-default-record traffic-policy-rule-schema)]}]})

(def keep-alive-schema
  {:name :keep-alive
   :type :record
   :fields [{:name :keep-alive
             :type :boolean
             :default true}]})
