(ns farbetter.mu.msg-xf
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.freedomdb.schemas :refer [DB]]
   [farbetter.mu.msgs :as msgs]
   [farbetter.mu.state :as state]
   [farbetter.mu.utils :as mu :refer
    [command-block Command CommandOrCommandBlock ConnId ConnState ProcType]]
   [farbetter.roe :as roe]
   [farbetter.roe.schemas :as rs :refer [AvroData]]
   [farbetter.utils :as u :refer
    [throw-far-error ByteArray #?@(:clj [go-safe inspect sym-map])]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [go-safe inspect sym-map]])))

;;;;;;;;;;;;;;;;;;;; Public Fns ;;;;;;;;;;;;;;;;;;;;

(s/defn handle-schema-rs :- [CommandOrCommandBlock]
  [db :- DB
   conn-id :- ConnId
   msg :- AvroData]
  (if (nil? (:json-schema msg))
    (do
      (errorf (str "Schema request failed. Peer %s did not have schema for "
                   "fingerprint `%s`") conn-id (:fingerprint msg))
      [])
    (let [{:keys [fingerprint json-schema]} msg
          _ (when (nil? json-schema))
          _ (when (nil? fingerprint)
              (throw-far-error "fingerprint is nil."
                               :illegal-argument :fingerprint-is-nil
                               (sym-map fingerprint msg)))
          rows (state/get-waiting-bytes db fingerprint)
          unordered-commands [[:delete-waiting-bytes fingerprint]
                              [:delete-schema-rq fingerprint]]
          init-ordered-commands [[:add-schema fingerprint json-schema]]
          ordered-commands (reduce (fn [acc row]
                                     (let [[conn-id bytes] row]
                                       (when (nil? conn-id)
                                         (throw-far-error
                                          "conn-id is nil"
                                          :illegal-argument :conn-id-is-nil
                                          (sym-map conn-id msg)))
                                       (conj acc [:inject-bytes conn-id bytes])))
                                   init-ordered-commands rows)
          block (apply command-block ordered-commands)]
      (conj unordered-commands block))))

(defn- make-schema-msg
  [db msg]
  (let [{:keys [fingerprint]} msg
        edn-schema (state/fingerprint->schema db fingerprint)
        json-schema (roe/edn-schema->json-schema edn-schema)]
    (sym-map fingerprint json-schema)))

(defn msg->commands [db conn-id msg-name msg proc-type addl-msg->op]
  (let [[state conn-type] (state/get-conn-state-and-type db conn-id)]
    (debugf "%s got %s msg from %s %s" proc-type msg-name conn-type conn-id)
    (case msg-name
      :keep-alive [] ;; do nothing on keepalive
      :schema-rq [[:send-msg conn-id :schema-rs (make-schema-msg db msg)]]
      :schema-rs (handle-schema-rs db conn-id msg)
      (if-let [op (addl-msg->op msg-name)]
        [[op conn-id msg]]
        (throw-far-error (str "Unknown msg name `" msg-name "`.")
                         :illegal-argument :unknown-msg
                         (sym-map msg-name conn-id proc-type msg))))))

(defn rq-time? [rq-time-ms]
  (or (not rq-time-ms)
      (> (u/get-current-time-ms)
         (+ rq-time-ms mu/schema-rq-interval-ms))))

(defn get-missing-schema-commands
  [db conn-id missing-fingerprint msg-fp enc-msg]
  (let [retry-fragment {:msg-id nil
                        :num-fragments 1
                        :fragment-num 0
                        :fingerprint msg-fp
                        :fragment-bytes enc-msg}
        fragment-bytes (roe/edn->avro-byte-array msgs/fragment-schema
                                                 retry-fragment)
        rq-time-ms (state/get-schema-rq-time-ms db missing-fingerprint)
        insert-bytes-cmd [:insert-bytes-waiting-for-schemas
                          missing-fingerprint conn-id fragment-bytes]]
    (if (rq-time? rq-time-ms)
      [(command-block insert-bytes-cmd
                      [:add-schema-rq missing-fingerprint]
                      [:send-msg conn-id :schema-rq
                       {:fingerprint missing-fingerprint}])]
      [insert-bytes-cmd])))

(defn- enc-msg->commands [db conn-id fingerprint enc-msg proc-type addl-msg->op]
  (if-let [writer-schema (state/fingerprint->schema db fingerprint)]
    (let [msg-name (:name writer-schema) ;; msgs always have a :name
          reader-schema-fp (state/name->fingerprint db msg-name)
          reader-schema (state/fingerprint->schema db reader-schema-fp)
          msg (roe/avro-byte-array->edn writer-schema reader-schema enc-msg)]
      (msg->commands db conn-id msg-name msg proc-type addl-msg->op))
    (get-missing-schema-commands db conn-id fingerprint
                                 fingerprint enc-msg)))

(defn- bytes->commands* [db conn-id bytes proc-type addl-msg->op]
  (let [fragment (roe/avro-byte-array->edn msgs/fragment-schema
                                           msgs/fragment-schema bytes)
        {:keys [msg-id num-fragments fragment-num
                fingerprint fragment-bytes]} fragment]
    (if (= 1 num-fragments)
      (enc-msg->commands db conn-id fingerprint fragment-bytes proc-type
                         addl-msg->op)
      (let [fragment-datas (state/get-fragment-bytes-seq db msg-id)
            num-fragment-datas (count fragment-datas)
            af-command [:add-fragment msg-id fragment-num fragment-bytes]]
        (if (= (dec num-fragments) num-fragment-datas)
          ;; Last frag, but not yet in db
          [(command-block af-command
                          [:inject-bytes conn-id bytes])]
          ;; Last frag and it's in the db
          (if (= num-fragments num-fragment-datas)
            (let [msg-data (u/concat-byte-arrays fragment-datas)]
              (mu/check-data-len (count msg-data))
              (conj (enc-msg->commands db conn-id fingerprint
                                       msg-data proc-type addl-msg->op)
                    [:delete-fragments msg-id]))
            ;; otherwise, just add the fragment
            [af-command]))))))

(defn- bytes->commands-connected [db conn-id bytes proc-type addl-msg->op]
  (let [peer-protocol-version (roe/avro-byte-array->edn :int :int bytes)]
    (if (= :gw proc-type)
      (if (= mu/protocol-version peer-protocol-version)
        [(command-block [:set-conn-state conn-id :protocol-negotiated]
                        [:send-bytes conn-id bytes])]
        [(command-block [:send-bytes conn-id (roe/edn->avro-byte-array :int 0)]
                        [:close-and-delete-conn conn-id])])
      (if (zero? peer-protocol-version)
        [[:close-and-delete-conn conn-id]]
        [(command-block [:set-conn-state conn-id :protocol-negotiated]
                        [:send-login conn-id])]))))

(s/defn bytes->commands :- [CommandOrCommandBlock]
  ([db :- DB
    conn-id :- ConnId
    bytes :- ByteArray
    proc-type :- ProcType]
   (bytes->commands db conn-id bytes proc-type {}))
  ([db :- DB
    conn-id :- ConnId
    bytes :- ByteArray
    proc-type :- ProcType
    addl-msg->op :- {s/Keyword s/Keyword}]
   (when (nil? conn-id)
     (throw-far-error "conn-id is nil"
                      :illegal-argument :conn-id-is-nil
                      (sym-map conn-id bytes proc-type)))
   (let [[state conn-type] (state/get-conn-state-and-type db conn-id)
         _ #?(:clj (debugf "%s got bytes %s from %s %s"
                           proc-type bytes conn-type conn-id)
              :cljs (debugf "%s got bytes (hash %s) from %s %s"
                            proc-type (hash bytes) conn-type conn-id))
         f (case state
             :start (throw-far-error
                     "Illegal state. Bytes rcvd while in :start state"
                     :execution-error :bytes-rcvd-before-connection
                     (sym-map state conn-id proc-type conn-type))
             :connected bytes->commands-connected
             bytes->commands*)]
     (if state
       (f db conn-id bytes proc-type addl-msg->op)
       []))))
