(ns farbetter.mu.gw-router
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.freedomdb :as fdb]
   [farbetter.freedomdb.schemas :refer [DB]]
   [farbetter.mu.utils :as mu :refer [ConnId MsgId RequestId SuccessFailureSeq]]
   [farbetter.roe.schemas :as rs :refer [AvroData AvroName]]
   [farbetter.utils :as u :refer
    [throw-far-error ByteArray #?@(:clj [go-safe inspect sym-map])]]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof warnf]]
   [schema.core :as s :include-macros true])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [go-safe inspect sym-map]])))

(def policy-load-interval-ms 3000)

(defn choose-rule [rules]
  (let [m (reduce (fn [acc rule]
                    (assoc acc rule (:weight rule)))
                  {} rules)]
    (u/weighted-rand m)))

(defn evaluate-rules [rules user-id rpc-rq-msg]
  (let [user-rules (filter (fn [rule]
                             (let [{:keys [users]} rule
                                   users-set (set users)]
                               (contains? users-set user-id))) rules)]
    (if (seq user-rules)
      (choose-rule user-rules)
      (let [star-rules (filter #(= (:users %) []) rules)]
        (when (seq star-rules)
          (choose-rule star-rules))))))

(defn- rq-id->str [rq-id]
  (-> rq-id
      (u/int-map->uuid)
      (u/uuid->hex-str)))

;; TODO: Rewrite this using a join?
(defn get-rule-for-rpc [db rpc-rq-msg user-id]
  (let [{:keys [service-api-version request-id]} rpc-rq-msg
        {:keys [service-name major-version]} service-api-version
        version-pairs (fdb/select
                       db :sis
                       {:fields [:minor-version :micro-version]
                        :where (sym-map service-name major-version)})]
    (if-not version-pairs
      (let [rq-id-str (rq-id->str request-id)
            reason (str "No SIs exist for " service-name
                        " version " major-version
                        ". Request-id: " (rq-id->str request-id))]
        (warnf reason)
        [:failure reason])
      (if-let [rules (fdb/select
                      db :traffic-policy-rules
                      {:where (sym-map service-name major-version)})]
        (let [vp-set (set version-pairs)

              rules (filter (fn [rule]
                              (let [{:keys [minor-version micro-version]} rule
                                    version-pair [minor-version micro-version]]
                                (contains? vp-set version-pair)))
                            rules)]
          [:success (evaluate-rules rules user-id rpc-rq-msg)])
        (let [rq-id-str (rq-id->str request-id)
              reason (str "No rules exist for " service-name
                          " version " major-version
                          ". Request-id: " (rq-id->str request-id))]
          [:failure reason])))))

(s/defn route-rpc-rq :- SuccessFailureSeq
  [db :- DB
   rpc-rq-msg :- AvroData]
  (let [{:keys [request-id user-id]} rpc-rq-msg
        ret (get-rule-for-rpc db rpc-rq-msg user-id)
        [status rule] ret]
    (if (= :success status)
      (let [{:keys [service-name major-version
                    minor-version micro-version]} rule
            si-conn-ids (fdb/select
                         db :sis
                         {:fields :si-conn-id
                          :where (sym-map service-name major-version
                                          minor-version micro-version)})]
        [:success (rand-nth si-conn-ids)])
      ret)))

(s/defn request-id->client-conn-id :- (s/maybe ConnId)
  [db :- DB
   request-id :- RequestId]
  (let [client-conn-id (fdb/select-one db :gw-rpcs
                                       {:fields :client-conn-id
                                        :where [:= :request-id request-id]})]
    client-conn-id))
