(ns farbetter.mu.gateway
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.mu.gw-router :as router]
   [farbetter.mu.msgs :as msgs]
   [farbetter.mu.proc :as proc]
   [farbetter.mu.state :as state]
   [farbetter.mu.msg-xf :as mxf]
   [farbetter.mu.utils :as mu :refer
    [command-block Command CommandOrCommandBlock ConnId MsgId RequestId]]
   [farbetter.pete :as pete]
   [farbetter.roe :as roe]
   [farbetter.roe.schemas :as rs :refer [AvroData AvroName]]
   [farbetter.utils :as u :refer
    [throw-far-error ByteArray #?@(:clj [go-safe inspect sym-map])]]
   [freedomdb.frontend :as fdb]
   [freedomdb.schemas :refer [DBType]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof warnf]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [go-safe inspect sym-map]])))

(declare make-on-connect)

(def rcv-chan-buf-size 1000)
(def rules-refresh-interval-ms (* 1000 2))
(def max-rpc-wait-ms (* 1000 20))

(defprotocol IGateway
  (serve [this])
  (stop [this]))

(defrecord Gateway [active?-atom db-atom repeater serve-fn stop-fn-atom
                    rcv-chan command-chan]
  IGateway
  (serve [this]
    (let [client-ws-handler (make-on-connect :cl rcv-chan command-chan)
          si-ws-handler (make-on-connect :si rcv-chan command-chan)]
      (reset! stop-fn-atom (serve-fn client-ws-handler si-ws-handler))))

  (stop [this]
    (@stop-fn-atom)
    (reset! active?-atom false)
    (pete/stop repeater)))

;;;;;;;;;;;;;;;;;;;; Helper fns ;;;;;;;;;;;;;;;;;;;;

(defn make-on-connect [conn-type rcv-chan command-chan]
  (fn on-connect [conn-id sender closer]
    (let [on-disconnect (fn [reason]
                          (let [commands (cond-> [[:delete-conn conn-id]]
                                           (= :si conn-type)
                                           (conj [:delete-si conn-id]))]
                            (debugf "GW on-disconnect. conn-id %s Reason: %s"
                                    conn-id reason)
                            (ca/put! command-chan commands)))
          on-error #(do (debugf "Error: %s" %)
                        (on-disconnect %))
          on-rcv (fn [data]
                   (ca/put! rcv-chan [conn-id data]))]
      (debugf "In gw on-connect. conn-id: %s" conn-id)
      (ca/put! command-chan [(command-block
                              [:add-conn conn-id sender closer conn-type]
                              [:set-conn-state conn-id :connected])])
      (sym-map on-rcv on-error on-disconnect))))

(s/defn handle-rpc-rq :- (s/maybe [CommandOrCommandBlock])
  [db :- DBType
   client-conn-id :- ConnId
   msg :- AvroData]
  (let [{:keys [request-id arg-schema-fp timeout-ms
                service-api-version fn-name]} msg]
    (if (state/fingerprint->schema db arg-schema-fp)
      (let [expiry-time-ms (+ timeout-ms (u/get-current-time-ms))
            [status v] (router/route-rpc-rq db msg)
            si-conn-id v]
        (if (= :success status)
          (do
            (debugf "Routing RPC %s to SI %s"
                    (u/int-map->hex-str request-id) si-conn-id)
            [(command-block [:insert-gw-rpc (sym-map client-conn-id si-conn-id
                                                     request-id expiry-time-ms)]
                            [:send-msg si-conn-id :rpc-rq msg])])
          (let [reason v
                rq-failed-msg (sym-map request-id reason)]
            (warnf reason)
            [[:send-msg client-conn-id :rpc-rs-failure rq-failed-msg]])))
      (let [enc-msg (roe/edn->avro-byte-array msgs/rpc-rq-schema msg)
            msg-fp (roe/edn-schema->fingerprint msgs/rpc-rq-schema)]
        (mxf/get-missing-schema-commands
         db client-conn-id arg-schema-fp msg-fp enc-msg)))))

(s/defn handle-rpc-rs-success :- (s/maybe [CommandOrCommandBlock])
  [db :- DBType
   si-conn-id :- ConnId
   msg :- AvroData]
  (let [{:keys [request-id return-schema-fp]} msg]
    (if (state/fingerprint->schema db return-schema-fp)
      (let [client-conn-id (router/request-id->client-conn-id db request-id)
            delete-rpc-command [:delete-gw-rpc request-id]]
        (if client-conn-id
          [(command-block delete-rpc-command
                          [:send-msg client-conn-id :rpc-rs-success msg])]
          [delete-rpc-command]))
      (let [enc-msg (roe/edn->avro-byte-array msgs/rpc-rs-success-schema msg)
            msg-fp (roe/edn-schema->fingerprint msgs/rpc-rs-success-schema)]
        (mxf/get-missing-schema-commands
         db si-conn-id return-schema-fp msg-fp enc-msg)))))

(s/defn handle-rpc-rs-failure
  [db :- DBType
   si-conn-id :- ConnId
   msg :- AvroData]
  (let [{:keys [request-id]} msg
        client-conn-id (router/request-id->client-conn-id db request-id)
        delete-rpc-command [:delete-gw-rpc request-id]]
    (if client-conn-id
      [(command-block delete-rpc-command
                      [:send-msg client-conn-id :rpc-rs-failure msg])]
      [delete-rpc-command])))

(defn make-si-login-handler [si-validator]
  (s/fn handle-si-login :- (s/maybe [CommandOrCommandBlock])
    [db :- DBType
     conn-id :- ConnId
     msg :- AvroData]
    (let [{:keys [access-key service-instance-version]} msg
          was-successful (si-validator access-key)
          send-msg-command [:send-msg conn-id :service-instance-login-rs
                            (sym-map was-successful)]]
      (debugf "In si-login-handler. was-successful: %s" was-successful)

      (if was-successful
        [(command-block [:insert-si (assoc service-instance-version
                                           :si-conn-id conn-id)]
                        [:set-conn-state conn-id :logged-in]
                        send-msg-command)]
        [(command-block [:close-and-delete-conn conn-id]
                        send-msg-command)]))))

(defn make-client-login-handler [user-validator]
  (s/fn handle-client-login :- (s/maybe [CommandOrCommandBlock])
    [db :- DBType
     conn-id :- ConnId
     msg :- AvroData]
    (let [{:keys [email password]} msg
          user-id (user-validator email password)
          was-successful (boolean user-id)
          send-msg-cmd [:send-msg conn-id :client-login-rs
                        (sym-map was-successful user-id)]
          cmds (if was-successful
                 [(command-block [:set-conn-state conn-id :logged-in]
                                 send-msg-cmd)]
                 [(command-block [:close-and-delete-conn conn-id]
                                 send-msg-cmd)])]
      (debugf "In :gw handle-client-login. commands: %s" cmds)
      cmds)))

(s/defn delete-gw-rpc :- DBType
  [db :- DBType
   request-id :- RequestId]
  (fdb/delete db :gw-rpcs [:= :request-id request-id]))

(s/defn delete-si :- DBType
  [db :- DBType
   conn-id :- ConnId]
  (fdb/delete db :sis [= :si-conn-id conn-id]))

(defn modify-db [db]
  [db :- DBType]
  (-> db
      (fdb/create-table :gw-rpcs {:client-conn-id {:type :any :indexed true}
                                  :si-conn-id {:type :any :indexed true}
                                  :request-id {:type :any :indexed true}
                                  :expiry-time-ms {:type mu/ms-time-type
                                                   :indexed true}})
      (fdb/create-table :sis {:si-conn-id {:type :any}
                              :service-name {:type :str1000}
                              :major-version {:type :int4}
                              :minor-version {:type :int4}
                              :micro-version {:type :int4}})
      (fdb/create-table :traffic-policy-rules
                        {:service-name {:type :str1000}
                         :major-version {:type :int4}
                         :minor-version {:type :int4}
                         :micro-version {:type :int4}
                         :users {:type :any}
                         :weight {:type :int4}})))

(s/defn insert-si :- DBType
  [db :- DBType
   row :- {:si-conn-id ConnId
           :service-name s/Str
           :major-version s/Num
           :minor-version s/Num
           :micro-version s/Num}]
  (fdb/insert db :sis row))

(defn get-gw-rpc-info [db request-id]
  (fdb/select-one db {:tables [:gw-rpcs]
                      :where [:= :request-id request-id]}))

(s/defn insert-gw-rpc :- DBType
  [db :- DBType
   row :- {:client-conn-id ConnId
           :si-conn-id ConnId
           :request-id RequestId
           :expiry-time-ms s/Num}]
  (fdb/insert db :gw-rpcs row))

(s/defn gc-gw-rpc :- (s/maybe [CommandOrCommandBlock])
  [db :- DBType]
  (let [now (u/get-current-time-ms)
        expired-rpcs (fdb/select db {:tables [:gw-rpcs]
                                     :where [:<= :expiry-time-ms now]})
        gc-rpc (fn [acc rpc]
                 (let [{:keys [client-conn-id request-id]} rpc
                       reason "RPC timed out. GW did not get response from SI."
                       msg (sym-map request-id reason)]
                   (-> acc
                       (conj (command-block [:delete-gw-rpc request-id]
                                            [:send-msg client-conn-id
                                             :rpc-rs-failure msg])))))]
    (reduce gc-rpc [] expired-rpcs)))

(defn insert-traffic-rule [db rule-row]
  (fdb/insert db :traffic-policy-rules rule-row))

(defn make-refresh-rules [rules-factory]
  (s/fn refresh-rules :- DBType
    [db :- DBType]
    (let [add-rows (fn [db]
                     (reduce insert-traffic-rule db (rules-factory)))]
      (-> db
          (fdb/delete :traffic-policy-rules)
          (add-rows)))))

;;;;;;;;;;;;;;;;;;;; Constructor ;;;;;;;;;;;;;;;;;;;;

(s/defn make-gateway :- (s/protocol IGateway)
  [user-validator :- (s/=> s/Any)
   si-validator :- (s/=> s/Any)
   rules-factory :- (s/=> s/Any)
   serve-fn :- (s/=> s/Any)]
  (let [db-atom (atom (state/make-db [] modify-db))
        repeater (pete/make-repeater)
        active?-atom (atom true)
        stop-fn-atom (atom (constantly nil))
        command-chan (ca/chan mu/chan-buf-size)
        rcv-chan (ca/chan mu/chan-buf-size)
        handle-client-login (make-client-login-handler user-validator)
        handle-si-login (make-si-login-handler si-validator)
        refresh-rules (make-refresh-rules rules-factory)
        addl-state-op->f (sym-map delete-gw-rpc delete-si insert-si
                                  insert-gw-rpc refresh-rules)
        addl-side-effect-op->f (sym-map handle-client-login handle-si-login
                                        handle-rpc-rq handle-rpc-rs-success
                                        handle-rpc-rs-failure gc-gw-rpc)
        addl-msg->op {:client-login-rq :handle-client-login
                      :service-instance-login-rq :handle-si-login
                      :rpc-rq :handle-rpc-rq
                      :rpc-rs-success :handle-rpc-rs-success
                      :rpc-rs-failure :handle-rpc-rs-failure}]
    (proc/start-processor active?-atom db-atom repeater command-chan rcv-chan
                          addl-side-effect-op->f addl-state-op->f
                          addl-msg->op :gw)
    (ca/put! command-chan [[:refresh-rules]])
    (pete/add-fn! repeater :refresh-rules
                  #(ca/put! command-chan [[:refresh-rules]])
                  rules-refresh-interval-ms)
    (pete/add-fn! repeater :gc-gw-rpc
                  #(do
                     (debugf "====== Enqueueing :gc-gw-rpc")
                     (ca/put! command-chan [[:gc-gw-rpc]]))
                  mu/gc-gw-rpc-interval-ms)
    (map->Gateway
     (sym-map active?-atom db-atom repeater serve-fn stop-fn-atom rcv-chan
              command-chan))))
