(ns farbetter.mu.client
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca
    :refer [#? (:clj go)]]
   [farbetter.mu.msgs :as msgs :refer [APISpec]]
   [farbetter.mu.msg-xf :as mxf]
   [farbetter.mu.proc :as proc]
   [farbetter.mu.state :as state]
   [farbetter.mu.transport :as tp]
   [farbetter.mu.utils :as mu :refer
    [command-block Command CommandOrCommandBlock ConnId RequestId UserId]]
   [farbetter.mu.websocket :as websocket]
   [farbetter.pete :as pete]
   [farbetter.roe :as roe]
   [farbetter.roe.schemas :refer [AvroData]]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [go-safe inspect sym-map])]]
   [freedomdb.frontend :as fdb]
   [freedomdb.schemas :refer [DBType]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [cljs.core.async.macros :refer [go]]
      [farbetter.utils :as u :refer [go-safe inspect sym-map]])))

(declare check-user-id)

(def bad-login-kw :bad-login)
(def rpc-check-interval-ms 500)

(defprotocol IClient
  (wait-for-login [this] "Returns a channel")
  (call-rpc
    [this service-name fn-name arg on-success on-failure]
    [this service-name fn-name arg on-success on-failure timeout-ms])
  (call-rpc-ch
    [this service-name fn-name arg]
    [this service-name fn-name arg timeout-ms])
  (collect-garbage [this]
    "Force garbage collection. Useful for testing.")
  (disconnect [this]
    "Disconnect the client from the network. Useful for testing")
  (stop [this]
    "Stop processing loop. Useful for testing."))

(defrecord Client [db-atom user-id-atom active?-atom command-chan
                   service-name->major-version repeater]
  IClient
  (wait-for-login [this]
    (go-safe
     (loop []
       (check-user-id user-id-atom)
       (when-not (state/get-logged-in-conn-ids @db-atom)
         (ca/<! (ca/timeout 5))
         (recur)))))

  (call-rpc [this service-name fn-name arg on-success on-failure]
    (call-rpc this service-name fn-name arg on-success on-failure
              mu/default-rpc-timeout-ms))

  (call-rpc [this service-name fn-name arg on-success on-failure timeout-ms]
    (check-user-id user-id-atom)
    (let [min-timeout (* 2 rpc-check-interval-ms)]
      (when (< timeout-ms min-timeout)
        (throw-far-error (str "timeout-ms cannot be shorter than "
                              min-timeout "ms.")
                         :illegal-argument :timeout-ms-too-short
                         (sym-map min-timeout timeout-ms
                                  service-name fn-name))))
    (try
      (let [db @db-atom
            user-id @user-id-atom
            request-id (-> (u/make-v1-uuid)
                           (u/uuid->int-map))
            major-version (service-name->major-version service-name)
            service-api-version (sym-map service-name major-version)
            arg-schema-name (state/make-rpc-schema-name
                             service-name fn-name :arg)
            arg-schema-fp (state/name->fingerprint db arg-schema-name)
            arg-schema (state/fingerprint->schema db arg-schema-fp)
            encoded-arg (roe/edn->avro-byte-array arg-schema arg)
            msg (sym-map service-api-version user-id request-id fn-name
                         arg-schema-fp encoded-arg timeout-ms)
            conn-id (state/get-gw-conn-id db)]
        (debugf ":cl sending %s RPC %s to GW."
                fn-name (u/int-map->hex-str request-id))
        (if conn-id
          (let [block (command-block [:add-client-rpc request-id on-success
                                      on-failure service-name fn-name
                                      timeout-ms]
                                     [:send-msg conn-id :rpc-rq msg])]
            (ca/put! command-chan [block]))
          (on-failure :not-logged-in))
        nil)
      (catch #?(:clj Exception :cljs :default) e
        (on-failure (u/get-exception-msg-and-stacktrace e)))))

  (call-rpc-ch [this service-name fn-name arg]
    (call-rpc-ch this service-name fn-name arg mu/default-rpc-timeout-ms))

  (call-rpc-ch [this service-name fn-name arg timeout-ms]
    (let [result-chan (ca/chan 1)
          on-success #(ca/put! result-chan [:success %])
          on-failure #(ca/put! result-chan [:failure %])]
      (call-rpc this service-name fn-name arg on-success on-failure timeout-ms)
      result-chan))

  (collect-garbage [this]
    (ca/put! command-chan [[:collect-garbage]]))

  (disconnect [this]
    (let [conn-ids (state/get-conn-ids @db-atom)]
      (doseq [conn-id conn-ids]
        (ca/put! command-chan [[:close-and-delete-conn conn-id]]))))

  (stop [this]
    (pete/stop repeater)
    (reset! active?-atom false)))

;;;;;;;;;;;;;;;;;;;; Helper fns ;;;;;;;;;;;;;;;;;;;;

(defn- check-user-id [user-id-atom]
  (when (= bad-login-kw @user-id-atom)
    (throw-far-error "Login failed due to bad credentials."
                     :execution-error :login-failed-bad-credentials {})))

(defn make-connect-to-gw [gw-urls-factory rcv-chan command-chan conn-factory]
  (s/fn client-connect-to-gw :- nil
    [db :- DBType]
    (go-safe  ;; TODO: Remove this and replace the <! below with a take!
     (let [[status gw-urls] (ca/<! (gw-urls-factory))]
       (case status
         :success (doseq [url gw-urls]
                    (tp/connect-to-gw url rcv-chan command-chan
                                      conn-factory :cl))
         :failure nil
         (throw-far-error
          (str "Bad status `" status
               "` returned from gw-urls-factory.")
          :execution-error :bad-status
          (sym-map status)))))
    nil))

(defn make-close-nli-commands [nli-conn-ids]
  (let [cad-commands (map (fn [conn-id]
                            [:close-and-delete-conn conn-id])
                          nli-conn-ids)
        block (-> (apply command-block cad-commands)
                  (conj [:connect-to-gw]))]
    [block]))

(s/defn start-connection-loop :- nil
  [active?-atom :- (s/atom s/Bool)
   db-atom :- (s/atom DBType)
   command-chan :- u/Channel]
  (go
    (try
      (while @active?-atom
        (let [db @db-atom
              li-conn-ids (state/get-logged-in-conn-ids db)
              commands (case (count li-conn-ids)
                         0 (let [nli-conn-ids (state/get-non-logged-in-conn-ids
                                               db)]
                             (if (pos? (count nli-conn-ids))
                               (when (every? #(state/timed-out? db %)
                                             nli-conn-ids)
                                 (make-close-nli-commands nli-conn-ids))
                               [[:connect-to-gw]]))
                         1 nil
                         [[:prune-conns]])]
          (when commands
            (ca/put! command-chan commands))
          (ca/<! (ca/timeout mu/max-loop-wait-ms))))
      (catch #?(:clj Exception :cljs :default) e
        (debugf "Got exception: %s" e)
        (u/log-exception e))))
  nil)

(defn make-handle-client-login-rs [user-id-atom]
  (s/fn handle-client-login-rs :- DBType
    [db :- DBType
     conn-id :- ConnId
     msg :- AvroData]
    (let [{:keys [was-successful user-id]} msg]
      (if was-successful
        (do
          (debugf "Login to %s was successful." conn-id)
          (reset! user-id-atom user-id)
          (state/set-conn-state db conn-id :logged-in))
        (throw-far-error "Client login failed due to invalid credentials."
                         :execution-error :login-failed-invalid-creds
                         (sym-map conn-id))))))

(s/defn handle-rpc-rs-success :- (s/maybe [CommandOrCommandBlock])
  [db :- DBType
   conn-id :- ConnId
   msg :- AvroData]
  (debugf "Client got rpc-rs. Request-id" (u/int-map->hex-str
                                           (:request-id msg)))
  (let [{:keys [request-id return-schema-fp encoded-return-value]} msg
        w-schema (state/fingerprint->schema db return-schema-fp)
        row (fdb/select-one db {:tables [:client-rpcs]
                                :where [:= :request-id request-id]})]
    (when row
      (let [{:keys [on-success service-name fn-name]} row
            return-schema-name (state/make-rpc-schema-name
                                service-name fn-name :return)]
        (if w-schema
          (let [return-schema-fp (state/name->fingerprint db return-schema-name)
                return-schema (state/fingerprint->schema db return-schema-fp)
                ret (roe/avro-byte-array->edn w-schema return-schema
                                              encoded-return-value)]
            (on-success ret)
            nil)
          (let [enc-msg (roe/edn->avro-byte-array
                         msgs/rpc-rs-success-schema msg)
                msg-fp (roe/edn-schema->fingerprint msgs/rpc-rs-success-schema)]
            (mxf/get-missing-schema-commands
             db conn-id return-schema-fp msg-fp enc-msg)))))))

(s/defn handle-rpc-rs-failure :- (s/maybe [CommandOrCommandBlock])
  [db :- DBType
   conn-id :- ConnId
   msg :- AvroData]
  (let [{:keys [request-id reason]} msg
        request-id-str (u/int-map->hex-str request-id)
        _ (debugf "Client got rq-failed msg. request-id: %s reason: %s"
                  request-id-str reason)
        on-failure (fdb/select-one db {:tables [:client-rpcs]
                                       :fields :on-failure
                                       :where [:= :request-id request-id]})]
    (if on-failure
      (on-failure {:request-id request-id-str :reason reason})
      (debugf "No on-failure handler found."))
    [[:delete-client-rpc request-id]]))

(s/defn delete-client-rpc :- DBType
  [db :- DBType
   request-id :- RequestId]
  (fdb/delete db :client-rpcs [:= :request-id request-id]))

(s/defn add-client-rpc :- DBType
  [db :- DBType
   request-id :- RequestId
   on-success :- (s/=> s/Any)
   on-failure :- (s/=> s/Any)
   service-name :- s/Str
   fn-name :- s/Str
   timeout-ms :- s/Num]
  (let [expiry-time-ms (+ timeout-ms (u/get-current-time-ms))]
    (fdb/insert db :client-rpcs
                (sym-map request-id on-success on-failure
                         service-name fn-name expiry-time-ms))))

(s/defn modify-db :- DBType
  [db :- DBType]
  (fdb/create-table db
                    :client-rpcs {:request-id {:type :any :indexed true}
                                  :on-success {:type :any :indexed false}
                                  :on-failure {:type :any :indexed false}
                                  :service-name {:type :str1000 :indexed false}
                                  :fn-name {:type :str1000 :indexed false}
                                  :expiry-time-ms {:type mu/ms-time-type
                                                   :indexed true}}))

(s/defn check-rpcs :- nil
  [db :- DBType
   command-chan :- u/Channel]
  (let [now (u/get-current-time-ms)
        expired-rpcs (fdb/select db {:tables [:client-rpcs]
                                     :where [:<= :expiry-time-ms now]})]
    (for [{:keys [request-id on-failure service-name fn-name]} expired-rpcs]
      (let [request-id (u/int-map->hex-str request-id)
            reason "RPC timed out"]
        (on-failure (sym-map reason request-id service-name fn-name))
        (ca/put! command-chan [[:delete-client-rpc request-id]])))))

(defn- check-apis [apis]
  (let [service-names (map :service-name apis)
        name-freqs (frequencies service-names)]
    (doseq [[name count] name-freqs]
      (when (> count 1)
        (throw-far-error (str "Duplicate service name `" name "`.")
                         :illegal-argument :duplicate-service-name
                         (sym-map name apis name-freqs service-names))))))

(defn- apis->service-name->major-version [apis]
  (reduce (fn [acc api]
            (let [{:keys [service-name major-version]} api]
              (assoc acc service-name major-version)))
          {} apis))

;;;;;;;;;;;;;;;;;;;; Constructor ;;;;;;;;;;;;;;;;;;;;

(s/defn make-client :- (s/protocol IClient)
  ;; TODO: Document parameters in docstring
  "Create a mu client.
   Parameters:
   - apis"
  ([apis :- [APISpec]
    email :- s/Str
    password :- s/Str
    gw-urls-factory :- (s/=> [s/Str])]
   (make-client apis email password gw-urls-factory
                websocket/make-client-websocket))
  ([apis :- [APISpec]
    email :- s/Str
    password :- s/Str
    gw-urls-factory :- (s/=> [s/Str])
    conn-factory :- (s/=> s/Any)]
   (check-apis apis)
   (let [db-atom (atom (state/make-db apis modify-db))
         user-id-atom (atom nil)
         active?-atom (atom true)
         command-chan (ca/chan mu/chan-buf-size)
         rcv-chan (ca/chan mu/chan-buf-size)
         service-name->major-version (apis->service-name->major-version apis)
         send-login (fn [db conn-id]
                      (let [msg (sym-map email password)]
                        [[:send-msg conn-id :client-login-rq msg]]))
         handle-client-login-rs (make-handle-client-login-rs user-id-atom)
         connect-to-gw (make-connect-to-gw gw-urls-factory rcv-chan command-chan
                                           conn-factory)
         addl-side-effect-op->f (sym-map handle-rpc-rs-success
                                         handle-rpc-rs-failure
                                         send-login connect-to-gw)
         addl-state-op-f (sym-map add-client-rpc delete-client-rpc
                                  handle-client-login-rs)
         addl-msg->op {:rpc-rs-success :handle-rpc-rs-success
                       :rpc-rs-failure :handle-rpc-rs-failure
                       :client-login-rs :handle-client-login-rs}
         repeater (pete/make-repeater)]
     (proc/start-processor active?-atom db-atom repeater command-chan rcv-chan
                           addl-side-effect-op->f addl-state-op-f
                           addl-msg->op :cl)
     (pete/add-fn! repeater :check-rpcs #(check-rpcs @db-atom command-chan)
                   rpc-check-interval-ms)
     (start-connection-loop active?-atom db-atom command-chan)
     (map->Client (sym-map db-atom user-id-atom active?-atom repeater
                           command-chan service-name->major-version)))))
