(ns farbetter.mu.service-instance
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.freedomdb.schemas :refer [DB]]
   [farbetter.mu.client :as mc]
   [farbetter.mu.msgs :as msgs :refer [APISpec FnMap]]
   [farbetter.mu.msg-xf :as mxf]
   [farbetter.mu.proc :as proc]
   [farbetter.mu.state :as state]
   [farbetter.mu.transport :as tp]
   [farbetter.mu.utils :as mu :refer
    [bad-login Command CommandOrCommandBlock ConnId ConnState Fingerprint
     ProcType RequestId UserId]]
   [farbetter.mu.websocket :as websocket]
   [farbetter.pete :as pete]
   [farbetter.roe :as roe]
   [farbetter.roe.schemas :as rs :refer [AvroData AvroSchema]]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [inspect sym-map])]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof tracef]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [inspect sym-map]])))

(def gw-conn-update-interval-ms (* 1000 1))

(declare get-gw-conn-id)

;;;;;;;;;;;;;;;;;;;; Protocol & Record ;;;;;;;;;;

(defprotocol IServiceInstance
  (start [this] "Start serving requests.")
  (stop [this] "Stop serving requests.")
  (update-conns [this] "Update gateway connections.")
  (wait-for-login
    [this]
    [this timeout-ms]
    "Wait until the SI has logged in to all the gateways. Returns a
     success/failure channel."))

(defrecord ServiceInstance [db-atom gw-urls-factory conn-factory rcv-chan
                            command-chan username password svc-fns api repeater
                            service-instance-version active?-atom
                            client-apis]
  IServiceInstance
  (start [this]
    (reset! active?-atom true)
    (pete/start repeater))

  (stop [this]
    (debugf ":si stop called")
    (proc/close-conns! @db-atom)
    (pete/stop repeater)
    (reset! active?-atom false)
    (ca/close! rcv-chan)
    (ca/close! command-chan))

  (update-conns [this]
    (u/go-sf ;; This is required because we do a <! below...
     (let [[status arg] (ca/<! (gw-urls-factory))
           _ (when-not (= :success status)
               (throw-far-error
                (str "gw-urls-factory call failed. reason: " arg)
                :execution-error :gw-urls-call-failed
                {:status status
                 :reason arg}))
           gw-urls (set arg)
           current-urls (set (state/get-conn-ids @db-atom))
           added-urls (clojure.set/difference gw-urls current-urls)
           deleted-urls (clojure.set/difference current-urls gw-urls)]
       (when (or (seq added-urls)
                 (seq deleted-urls))
         (debugf "Update conns:\n  cur: %s\n  added: %s\n  deleted: %s"
                 current-urls added-urls deleted-urls))
       (doseq [url added-urls]
         (tp/connect-to-gw url rcv-chan command-chan db-atom conn-factory :si))
       (doseq [url deleted-urls]
         (ca/put! command-chan [[:close-and-delete-conn url]])))))

  (wait-for-login [this]
    (wait-for-login this mu/default-login-wait-ms))

  (wait-for-login [this timeout-ms]
    (u/go-sf
     (let [expiry-ms (+ (u/get-current-time-ms) timeout-ms)]
       (loop []
         (let [[status arg]  (ca/<! (gw-urls-factory))
               _ (when-not (= :success status)
                   (throw-far-error
                    (str "gw-urls-factory call failed. reason: " arg)
                    :execution-error :gw-urls-call-failed {:reason arg}))
               gw-urls (set arg)
               logged-in-conn-ids (set (state/get-logged-in-conn-ids
                                        @db-atom))]
           (if (= gw-urls logged-in-conn-ids)
             gw-urls
             (if (>= (u/get-current-time-ms) expiry-ms)
               (throw-far-error "Timed out waiting for SI login"
                                :execution-error :timed-out-waiting-for-si-login
                                (sym-map timeout-ms))
               (do
                 (ca/<! (ca/timeout 5))
                 (recur)))))))))

  mc/IRPCClient
  (call-rpc [this service-name fn-name arg]
    (mc/call-rpc this service-name fn-name arg {}))
  (call-rpc [this service-name fn-name arg opts]
    (when-not (seq client-apis)
      (throw-far-error
       (str "SI RPC is impossible, since no client APIs were defined at SI "
            "construction.")
       :execution-error :no-client-apis-defined {}))
    (mc/call-rpc* client-apis service-name fn-name arg command-chan
                  rcv-chan db-atom :si #(wait-for-login this %)
                  #(get-gw-conn-id @db-atom) opts)))

;;;;;;;;;;;;;;;;;;;; Helper fns ;;;;;;;;;;;;;;;;;;;;

(defn get-gw-conn-id [db]
  (let [li-conn-ids (state/get-logged-in-conn-ids db)]
    (when-not (nil? li-conn-ids)
      (rand-nth li-conn-ids))))

(s/defn handle-si-login-rs :- (s/maybe [CommandOrCommandBlock])
  [db :- DB
   conn-id :- ConnId
   msg :- AvroData]
  (let [{:keys [was-successful user-id]} msg]
    (if was-successful
      [[:set-conn-state conn-id :logged-in]
       [:set-user-id user-id]]
      (throw-far-error "SI login failed due to incorrect credentials."
                       :execution-error :bad-si-login-credentials {}))))

(defn stringify-map [m]
  (reduce-kv (fn [acc k v]
               (assoc acc (str k) (str v)))
             {} m))

(s/defn execute-fn :- (s/eq nil)
  [db :- DB
   conn-id :- ConnId
   request-id :- RequestId
   user-id :- UserId
   timeout-ms :- s/Num
   return-schema-fp :- Fingerprint
   fn-name :- s/Str
   f :- (s/=> s/Any)
   arg :- AvroData
   command-chan :- u/Channel]
  (when (nil? conn-id)
    (throw-far-error "conn-id is nil."
                     :illegal-argument :conn-id-is-nil
                     (sym-map request-id conn-id fn-name)))
  (when (nil? request-id)
    (throw-far-error "request-id is nil."
                     :illegal-argument :request-id-is-nil
                     (sym-map request-id conn-id fn-name)))
  (when (nil? return-schema-fp)
    (throw-far-error "return-schema-fp is nil."
                     :illegal-argument :return-schema-fp-is-nil
                     (sym-map return-schema-fp request-id conn-id fn-name)))
  (when (nil? f)
    (throw-far-error "f is nil."
                     :illegal-argument :f-id-is-nil
                     (sym-map f request-id conn-id fn-name)))
  (u/go-sf
   (let [request-id-str (u/int-map->hex-str request-id)]
     (debugf "Entering :si execute-fn. fn-name: %s request-id: %s"
             fn-name request-id-str)
     (try
       (let [metadata (sym-map user-id request-id timeout-ms)
             ret-or-ch (f arg metadata)
             ret (if (u/channel? ret-or-ch)
                   (let [[status result] (ca/<! ret-or-ch)]
                     (if (= :success status)
                       result
                       (throw-far-error
                        (str "Service fn failed.")
                        :execution-error :rpc-failed
                        (sym-map status result fn-name arg request-id-str))))
                   ret-or-ch)
             return-schema (state/fingerprint->schema db return-schema-fp)
             encoded-return-value (roe/edn->avro-byte-array return-schema ret)
             ret-msg (sym-map request-id return-schema-fp encoded-return-value)]
         (ca/put! command-chan [[:send-msg conn-id :rpc-rs-success ret-msg]])
         (tracef "Exiting service-instance/execute-fn for %s" fn-name))
       (catch #?(:clj Exception :cljs :default) e
         (let [exception-msg (u/get-exception-msg e)
               stacktrace (u/get-exception-stacktrace e)
               error-map (ex-data e)
               type (str (:type error-map))
               subtype (str (:subtype error-map))
               error-map (->  error-map
                              (dissoc :type :subtype)
                              (stringify-map))
               reason (sym-map type subtype error-map exception-msg
                               stacktrace request-id-str fn-name)
               msg (sym-map request-id reason)]
           (when-not mc/*mute-errors*
             (errorf (str reason)))
           (ca/put! command-chan [[:send-msg conn-id :rpc-rs-failure msg]]))))))
  nil)

(defn make-rpc-rq-handler [service-instance-version svc-fns command-chan]
  (s/fn handle-rpc-rq :- (s/maybe [CommandOrCommandBlock])
    [db :- DB
     conn-id :- ConnId
     msg :- AvroData]
    (let [{:keys [service-name]} service-instance-version
          {:keys [request-id user-id timeout-ms fn-name arg-schema-fp
                  encoded-arg]} msg
          arg-schema-name (state/make-rpc-schema-name service-name fn-name :arg)
          return-schema-name (state/make-rpc-schema-name
                              service-name fn-name :return)
          w-schema (state/fingerprint->schema db arg-schema-fp)]
      (if w-schema
        (let [arg-schema-fp (state/name->fingerprint db arg-schema-name)
              return-schema-fp (state/name->fingerprint db return-schema-name)
              arg-schema (state/fingerprint->schema db arg-schema-fp)
              arg (roe/avro-byte-array->edn w-schema arg-schema encoded-arg)
              f (svc-fns fn-name)]
          (execute-fn db conn-id request-id user-id timeout-ms return-schema-fp
                      fn-name f arg command-chan))
        (let [enc-msg (roe/edn->avro-byte-array msgs/rpc-rq-schema msg)
              msg-fp (roe/edn-schema->fingerprint msgs/rpc-rq-schema)
              cmds (mxf/get-missing-schema-commands
                    db conn-id arg-schema-fp msg-fp enc-msg)]
          cmds)))))

;;;;;;;;;;;;;;;;;;;; Constructor ;;;;;;;;;;;;;;;;;;;;

(s/defn make-service-instance :- (s/protocol IServiceInstance)
  "Create a mu service instance.
   Arguments:
   - api - Map describing the API of the service instance. Required keys:
        - :service-name - String. Name of the service.
        - :major-version - Integer. Major version number of the API.
        - :fns - Map of fn-name keys to fns of one arg
   - svc-fns - Map of fn-name keys to fn values.
   - minor-version - Integer. Minor version of this implementation.
   - micro-version - Integer. Micro version of this implementation.
   - username - String. Service instance username. Used for GW login.
   - password - String. Service instance password. Used for GW login.
   - opts - A optional map of options. Valid options:
        - :client-apis - A sequence of other service APIs that this SI will be
            able to call. Each API should conform to the APISpec schema.
        - :gw-urls-factory - Fn of no arguments. Should return a success/failure
             channel. If successful, the second argument should be a sequence of
             valid gateway URLs (as strings).
        - :conn-factory - A fn to construct a connection. If not
             specfied, websockets will be used. If specified, the fn should
             accept these arguments (all are required):
                 - url - The URL to connect to. String.
                 - on-connect - A fn of no arguments to be called on connection.
                 - on-disconnect - A fn of one argument (reason) to be called
                     on disconnection.
                 - on-error - A fn of one argument (reason); called on error.
                 - on-rcv - A fn of one argument (data); called when data
                     is received.
             The fn should return a map with two keys:
                 - sender - A fn of one argument (data) that can be called to
                     send data over the connection.
                 - closer - A fn of no arguments that can be called to close the
                     connection."

  ([api :- APISpec
    svc-fns :- FnMap
    minor-version :- s/Int
    micro-version :- s/Int
    username :- s/Str
    password :- s/Str]
   (make-service-instance api svc-fns minor-version micro-version
                          username password {}))
  ([api :- APISpec
    svc-fns :- FnMap
    minor-version :- s/Int
    micro-version :- s/Int
    username :- s/Str
    password :- s/Str
    opts :- {(s/optional-key :client-apis) [APISpec]
             (s/optional-key :gw-urls-factory) (s/=> [s/Str])
             (s/optional-key :conn-factory) (s/=> s/Any)}]
   (let [{:keys [client-apis gw-urls-factory conn-factory]
          :or {client-apis []
               gw-urls-factory #?(:clj mu/get-gw-urls
                                  :cljs #(throw-far-error
                                          "CLJS must specify a gw-urls-factory"
                                          :execution-error :no-gw-urls-factory
                                          (sym-map opts)))
               conn-factory websocket/make-si-websocket}} opts
         {:keys [service-name major-version fns]} api
         api-fn-names (set (keys fns))
         svc-fn-names (set (keys svc-fns))
         missing-fns (clojure.set/difference api-fn-names svc-fn-names)
         _ (when (seq missing-fns)
             (throw-far-error (str "Missing svc-fn(s):" missing-fns)
                              :illegal-argument :missing-fns
                              (sym-map api svc-fns missing-fns)))
         command-chan (ca/chan mu/chan-buf-size)
         rcv-chan (ca/chan mu/chan-buf-size)
         repeater (pete/make-repeater)
         service-instance-version (sym-map service-name major-version
                                           minor-version micro-version)
         db-atom (atom (state/make-db (conj client-apis api) mc/modify-db))
         active?-atom (atom true)
         login-failed?-atom (atom false)
         send-login (fn [db conn-id]
                      (let [msg (sym-map username password
                                         service-instance-version)]
                        [[:send-msg conn-id :service-instance-login-rq msg]]))
         handle-rpc-rq (make-rpc-rq-handler service-instance-version svc-fns
                                            command-chan)
         rpc-rs-handlers {:handle-rpc-rs-failure mc/handle-rpc-rs-failure
                          :handle-rpc-rs-success mc/handle-rpc-rs-success}
         addl-side-effect-op->f (merge rpc-rs-handlers
                                       (sym-map handle-rpc-rq
                                                handle-si-login-rs
                                                send-login))
         addl-state-op->f {:add-client-rpc mc/add-client-rpc
                           :delete-client-rpc mc/delete-client-rpc
                           :set-user-id mc/set-user-id}
         addl-msg->op {:rpc-rq :handle-rpc-rq
                       :rpc-rs-success :handle-rpc-rs-success
                       :rpc-rs-failure :handle-rpc-rs-failure
                       :service-instance-login-rs :handle-si-login-rs}
         params (sym-map db-atom gw-urls-factory conn-factory repeater rcv-chan
                         username password svc-fns api service-instance-version
                         active?-atom command-chan login-failed?-atom
                         client-apis)
         si (map->ServiceInstance params)]
     (proc/start-processor active?-atom db-atom repeater command-chan
                           rcv-chan addl-side-effect-op->f addl-state-op->f
                           addl-msg->op :si)
     ;; Explicitly call update-conns the first time so we don't have
     ;; to wait for pete to schedule it
     (update-conns si)
     (pete/add-fn! repeater :update-conns #(update-conns si)
                   gw-conn-update-interval-ms)

     (start si)
     si)))
