(ns farbetter.mu.proc
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.mu.msgs :as msgs]
   [farbetter.mu.state :as state]
   [farbetter.mu.utils :as mu :refer [Channel Command OpToF ProcType]]
   [farbetter.mu.msg-xf :as mx]
   [farbetter.pete :as pete]
   [farbetter.roe :as roe]
   [farbetter.utils :as u :refer
    [throw-far-error ByteArray #?@(:clj [go-safe inspect sym-map])]]
   [freedomdb.frontend :as fdb]
   [freedomdb.schemas :refer [DBType]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [go-safe inspect sym-map]])))

(def gc-interval-ms (* 1000 60))
(def do-send-interval-ms 1)
(def rcv-chan-timeout-ms 500)

;;;;;;;;;;;;;;;;;;;; Public Fns ;;;;;;;;;;;;;;;;;;;;

(defn- make-command-proc [rcv-chan addl-op->f proc-type]
  (fn [db command]
    (if (= :inject-bytes (first command))
      (do
        (ca/put! rcv-chan (rest command))
        db)
      (state/process-command db command addl-op->f))))

(s/defn process-commands :- DBType
  [db :- DBType
   commands :- [Command]
   addl-op->f :- OpToF
   rcv-chan :- Channel
   proc-type :- ProcType]
  (reduce (make-command-proc rcv-chan addl-op->f proc-type)
          db commands))

(s/defn start-data-processing-loop :- nil
  [active?-atom :- (s/atom s/Bool)
   db-atom :- (s/atom DBType)
   rcv-chan :- Channel
   proc-type :- ProcType
   addl-op->f :- OpToF]
  (go-safe
   (while @active?-atom
     (try
       (let [[v ch] (ca/alts! [rcv-chan (ca/timeout rcv-chan-timeout-ms)])]
         (when (= rcv-chan ch)
           (let [[conn-id bytes] v
                 process (fn [db]
                           (let [commands (mx/bytes->commands
                                           db conn-id bytes proc-type)]
                             (process-commands db commands addl-op->f
                                               rcv-chan proc-type)))]
             (swap! db-atom process))))
       (catch #?(:clj Exception :cljs :default) e
         (u/log-exception e)))))
  nil)

(s/defn collect-garbage! :- nil
  [db-atom :- (s/atom DBType)
   addl-gc :- (s/=> DBType DBType)]
  (swap! db-atom #(-> %
                      (state/gc-fragments)
                      (state/gc-schema-rqs)
                      (state/gc-msgs-waiting)
                      (addl-gc)))
  nil)

(defn- make-keep-alive-bytes []
  (let [ka-msg {:keep-alive true}
        fragment-data (roe/edn->avro-byte-array msgs/keep-alive-schema ka-msg)
        fingerprint (roe/edn-schema->fingerprint msgs/keep-alive-schema)
        fragment (sym-map fingerprint fragment-data)
        bytes (roe/edn->avro-byte-array msgs/fragment-schema fragment)]
    bytes))

(def keep-alive-bytes (make-keep-alive-bytes))

(s/defn keep-alive! :- nil
  [db :- DBType
   proc-type :- ProcType]
  (let [rows (fdb/select db {:tables [:conns]
                            :fields [:sender :conn-id]
                             :where [:= :state :logged-in]})]
    (doseq [[sender conn-id] rows]
      (sender keep-alive-bytes))))

(s/defn do-send! :- nil
  [db-atom :- (s/atom DBType)]
  (when-let [rows (fdb/select @db-atom {:tables [:to-send]})]
    (doseq [{:keys [id conn-id bytes]} rows]
      (when-let [sender (fdb/select-one @db-atom
                                        {:tables [:conns]
                                         :fields :sender
                                         :where [:= :conn-id conn-id]})]
        (sender bytes)
        (swap! db-atom fdb/delete :to-send [:= :id id])))))

(s/defn start-procs :- nil
  [repeater active?-atom db-atom rcv-chan proc-type addl-op->f addl-gc]
  (debugf "### Starting procs for %s" proc-type)
  (start-data-processing-loop active?-atom db-atom rcv-chan proc-type
                              addl-op->f)
  (pete/add-fn repeater :do-send #(do-send! db-atom) do-send-interval-ms)
  (pete/add-fn repeater :keep-alive #(keep-alive! @db-atom proc-type)
               mu/keep-alive-interval-ms)
  (pete/add-fn repeater :gc #(collect-garbage! db-atom addl-gc) gc-interval-ms))
