(ns farbetter.mu.utils
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.mu.msgs :as msgs]
   [farbetter.roe :as roe]
   [farbetter.roe.schemas :as rs :refer [AvroSchema]]
   [farbetter.utils :as u :refer
    [throw-far-error ByteArray #?@(:clj [inspect sym-map])]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [inspect sym-map]])))

(def valid-proc-types #{:si :cl :gw})
(def protocol-version 1)
(def valid-conn-states #{:start :connected :protocol-negotiated :logged-in})
(def max-data-bytes  1e7)
(def max-fragment-bytes (* 1000 32))

;; Time values
(def default-rpc-timeout-ms (* 1000 10))
(def keep-alive-interval-ms (* 1000 20))
(def gc-interval-ms (* 1000 10))
(def gc-gw-rpc-interval-ms (* 1000 1))
(def max-loop-wait-ms 100)
(def schema-rq-interval-ms 1000)
(def add-conn-timeout-ms 100)
(def get-sender-timeout-ms 1000)
(def check-waiting-sends-interval-ms 100)
(def default-login-wait-ms 10000)

(def chan-buf-size 1000)
(def client-endpoint-path "client-ws")
(def si-endpoint-path "service-ws")
(def msg-schemas
  [msgs/client-login-rq-schema msgs/client-login-rs-schema
   msgs/keep-alive-schema msgs/rpc-rq-schema msgs/rpc-rs-failure-schema
   msgs/rpc-rs-success-schema msgs/service-instance-login-rq-schema
   msgs/schema-rq-schema msgs/schema-rs-schema
   msgs/service-instance-login-rs-schema])

(def bad-login :bad-login)

;;;;;;;;;;;;;;;;;;;; Schemas ;;;;;;;;;;;;;;;;;;;;

(def Command [(s/one s/Keyword "op") s/Any])
(def CommandBlock [Command])
(def CommandOrCommandBlock (s/if #(keyword? (first %))
                             Command
                             CommandBlock))
(def ConnState (s/pred valid-conn-states))
(def Fingerprint (s/pred u/long?))
(def Timestamp (s/pred u/long?))
(def Fragment {(s/optional-key :msg-id) s/Any
               (s/required-key :num-fragments) s/Num
               (s/required-key :fragment-num) s/Num
               (s/required-key :fingerprint) Fingerprint
               (s/required-key :fragment-bytes) ByteArray})
(def UUIDMap {(s/required-key :high) s/Int
              (s/required-key :midh) s/Int
              (s/required-key :midl) s/Int
              (s/required-key :low) s/Int})
(def ProcType (s/pred valid-proc-types))
(def OpToF {s/Keyword (s/=> s/Any)})
(def MsgId UUIDMap)
(def RequestId UUIDMap)
(def UserId UUIDMap)
(def ConnId s/Str)
(def SuccessFailureSeq [(s/one (s/enum :success :failure) "status")
                        (s/one s/Any "value")])
(def APIInfo {:service-name s/Str
              :major-version s/Num
              :fn-name s/Str
              :arg-schema AvroSchema
              :return-schema AvroSchema})

(def Metadata {:request-id RequestId
               :user-id UserId
               :timeout-ms s/Num})

(def gw-urls-schema {:type :array
                     :items :string})

;;;;;;;;;;;;;;;;;;;; Public Fns ;;;;;;;;;;;;;;;;;;;;

(defn check-data-len [data-len]
  (when (> data-len max-data-bytes)
    (throw-far-error
     (str "Msg is too large. Msgs must be less than " max-data-bytes
          " bytes when encoded.")
     :illegal-argument :msg-too-large (sym-map data-len))))

(defn command-block [& commands]
  (vec commands))

(defn parse-urls-str [urls-str]
  (let [urls (clojure.string/split urls-str #",")]
    (if (and (seq urls)
             (every? #(re-matches #"wss?:\S+" %) urls))
      urls
      (throw-far-error
       (str "Failed to parse urls from TEST_GW_URLS: " urls-str)
       :execution-error :could-not-parse-urls (sym-map urls urls-str)))))

;; Default get-gw-urls depends on env vars, so is clj only.
#?(:clj
   (defn get-gw-urls []
     (u/go-sf
      (if-let [gw-urls-str (u/get-env-var "TEST_GW_URLS" String nil)]
        (parse-urls-str gw-urls-str)
        (if-let [factory-url (u/get-env-var "GET_GW_URLS_URL" String nil)]
          (let [ch (u/http-get factory-url)
                {:keys [status body error]} (ca/<! ch)]
            (if (= 200 status)
              (roe/avro-b64-string->edn gw-urls-schema gw-urls-schema
                                        body)
              (throw-far-error "Failed to get gateway URLS"
                               :execution-error :get-gw-urls-failed
                               (sym-map status body error))))
          (throw-far-error
           (str "No way to get GW URLS. GET_GW_URLS_URL is not set. "
                "TEST_GW_URLS is not set.")
           :execution-error :no-way-to-get-gw-urls {}))))))
