(ns farbetter.mu.transport
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [inspect sym-map])]]
   [farbetter.freedomdb.schemas :refer [DB]]
   [farbetter.mu.state :as state]
   [farbetter.mu.utils :as mu :refer
    [command-block CommandOrCommandBlock ConnId ProcType]]
   [farbetter.roe :as roe]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [inspect sym-map]])))

(def protocol-bytes (roe/edn->avro-byte-array :int mu/protocol-version))

(defn make-on-rcv [conn-id rcv-chan db-atom]
  (when (nil? conn-id)
    (throw-far-error "conn-id is nil."
                     :illegal-argument :conn-id-is-nil
                     (sym-map conn-id)))
  (fn [data]
    (u/go-sf
     (loop []
       (let [[state _] (state/get-conn-state-and-type
                        @db-atom conn-id)]
         (when (or (not state)
                   (= :start state))
           (ca/<! (ca/timeout 10))
           (recur))))
     (ca/put! rcv-chan [conn-id data])
     nil)))

(s/defn connect-to-gw :- nil
  [conn-id :- ConnId
   rcv-chan :- u/Channel
   command-chan :- u/Channel
   db-atom :- (s/atom DB)
   conn-factory :- (s/=> s/Any)
   proc-type :- ProcType]
  (when (nil? conn-id)
    (throw-far-error "conn-id is nil."
                     :illegal-argument :conn-id-is-nil
                     (sym-map conn-id proc-type)))
  (debugf "In %s connect-to-gw. conn-id: %s" proc-type conn-id)
  (u/go-sf
   (let [conn-added-ch (ca/chan 1)
         got-sender-closer-ch (ca/chan 1)
         _ (ca/put! command-chan [(command-block
                                   [:add-conn conn-id nil nil :gw]
                                   [:notify conn-added-ch])])
         on-connect #(u/go-sf
                      (debugf "In %s on-connect. conn-id: %s"
                              proc-type conn-id)
                      (let [timeout-ch (ca/timeout mu/add-conn-timeout-ms)
                            [v ch] (ca/alts! [conn-added-ch timeout-ch])]
                        (when (= conn-added-ch ch)
                          (let [timeout-ch (ca/timeout mu/get-sender-timeout-ms)
                                [v ch] (ca/alts! [got-sender-closer-ch
                                                  timeout-ch])
                                [sender closer] v
                                block [(command-block
                                        [:set-sender-closer conn-id
                                         sender closer]
                                        [:set-conn-state conn-id :connected]
                                        [:send-bytes conn-id protocol-bytes])]]
                            (if (= timeout-ch ch)
                              (ca/>! command-chan [[:close-and-delete conn-id]])
                              (ca/put! command-chan block)))))
                      nil)

         on-disconnect #(do
                          (infof (str "Connection to " conn-id " closed. "
                                      "Reason: " %))
                          (ca/put! command-chan
                                   ;; TODO:
                                   ;;:close-and-delete works with gniazdo. Does
                                   ;; it work with node/socket-rocket/browser?
                                   [[:close-and-delete-conn conn-id]]))
         on-error #(do (debugf "Error: %s" %)
                       (on-disconnect %))
         on-rcv (make-on-rcv conn-id rcv-chan db-atom)
         {:keys [sender closer]} (conn-factory conn-id on-connect on-disconnect
                                               on-error on-rcv)]
     (if (and sender closer)
       (ca/put! got-sender-closer-ch [sender closer])
       (ca/put! command-chan [[:delete-conn conn-id]]))))
  nil)
