(ns farbetter.mu.proc
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.freedomdb :as fdb]
   [farbetter.freedomdb.schemas :refer [DB]]
   [farbetter.mu.msgs :as msgs]
   [farbetter.mu.state :as state]
   [farbetter.mu.utils :as mu :refer
    [Command CommandOrCommandBlock ConnId OpToF ProcType]]
   [farbetter.mu.msg-xf :as mxf]
   [farbetter.pete :as pete]
   [farbetter.roe :as roe]
   [farbetter.utils :as u :refer
    [throw-far-error ByteArray #?@(:clj [inspect sym-map])]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [inspect sym-map]])))

(declare process-command-or-block)

(def command-debug-blacklist
  #{:collect-garbage :gc-client-rpcs :gc-gw-rpc :gc-fragments :gc-msgs-waiting
    :gc-schema-rqs :refresh-rules :update-client-conns})
(def command-debug-expanded
  #{:set-conn-state})

;; Get rid of jetty logging msgs
(timbre/merge-config!
 {:ns-blacklist ["org.eclipse.jetty.*"]})

;;;;;;;;;;;;;;;;;;;; Public Fns ;;;;;;;;;;;;;;;;;;;;

(defn- make-keep-alive-bytes []
  (let [ka-msg {:keep-alive true}
        fragment-bytes (roe/edn->avro-byte-array msgs/keep-alive-schema ka-msg)
        fingerprint (roe/edn-schema->fingerprint msgs/keep-alive-schema)
        fragment (sym-map fingerprint fragment-bytes)
        bytes (roe/edn->avro-byte-array msgs/fragment-schema fragment)]
    bytes))

(def keep-alive-bytes (make-keep-alive-bytes))

(s/defn close-and-delete-conn :- (s/maybe [CommandOrCommandBlock])
  [db :- DB
   conn-id :- ConnId]
  (when-let [closer (state/get-closer db conn-id)]
    (try
      (debugf "Calling closer for %s" conn-id)
      (closer)
      (catch #?(:clj Exception :cljs :default) e
        (errorf "Closer threw an exception: %s" e)
        db)))
  [[:delete-conn conn-id]])

;; This is for testing mostly; it is not a command
(s/defn close-conns! :- nil
  [db :- DB]
  (doseq [conn (state/get-conns db)]
    (when-let [closer (:closer conn)]
      (debugf "Calling closer for %s" (:conn-id conn))
      (closer))))

(s/defn prune-conns :- (s/maybe [CommandOrCommandBlock])
  "Delete all :logged-in connections except one."
  [db :- DB]
  (let [li-conn-ids (state/get-logged-in-conn-ids db)]
    (for [conn-id (rest li-conn-ids)]
      [[:close-and-delete-conn conn-id]])))

(s/defn collect-garbage :- (s/maybe [CommandOrCommandBlock])
  [db :- DB]
  (concat [[:gc-fragments]
           [:gc-schema-rqs]
           [:gc-msgs-waiting]]))

(s/defn notify :- (s/maybe [CommandOrCommandBlock])
  [db :- DB
   notify-channel :- u/Channel]
  (ca/put! notify-channel true)
  nil)

(s/defn do-send :- nil
  [sender bytes-seq msg-name conn-type conn-id proc-type]
  (when-not (= "keep-alive" msg-name)
    (debugf "%s sending %s to %s %s"
            proc-type msg-name conn-type conn-id))
  (doseq [bytes bytes-seq]
    (sender bytes)))

(defn- msg->enc-fragments [db msg-name msg]
  (let [fingerprint (state/name->fingerprint db msg-name)
        _ (when-not fingerprint
            (throw-far-error (str "Unknown msg-name: " msg-name)
                             :execution-error :unknown-msg-name
                             (sym-map msg-name msg)))
        schema (state/fingerprint->schema db fingerprint)
        bytes (roe/edn->avro-byte-array schema msg)
        _ (mu/check-data-len (count bytes))]
    (state/bytes->enc-fragments fingerprint bytes)))

(defn- send-q [sender q conn-id proc-type]
  (debugf "%s dequeuing bytes to send to %s"
          proc-type conn-id)
  (loop [q q]
    (when-let [info (peek q)]
      (let [{:keys [bytes-seq msg-name conn-type conn-id]} info]
        (do-send sender bytes-seq msg-name conn-type conn-id proc-type)
        (recur (pop q))))))

(def Excuse (s/enum :not-ready :not-present))
(def Sender (s/=> s/Any))
(def SenderOrExcuse (s/if keyword?
                      Excuse
                      Sender))

(s/defn get-sender-or-excuse :- SenderOrExcuse
  [db :- DB
   conn-id :- ConnId
   proc-type :- ProcType]
  (let [conn (state/get-conn db conn-id)
        {:keys [sender state]} conn
        s-o-e (if conn
                (if (and sender
                         (not= :start state))
                  sender
                  :not-ready)
                :not-present)]
    s-o-e))

(s/defn check-waiting-sends :- nil
  [db conn-id->waiting-q-atom proc-type]
  (let [rm-q #(swap! conn-id->waiting-q-atom dissoc %)]
    (doseq [[conn-id q] @conn-id->waiting-q-atom]
      (let [sender-or-excuse (get-sender-or-excuse db conn-id proc-type)]
        (cond
          (= :not-ready sender-or-excuse)
          nil

          ;; If there is no longer a row in the db (because the conn
          ;; timed out), just drop all waiting traffic in the queue.
          (= :not-present sender-or-excuse)
          (do
            (debugf (str "%s dropping enqueued traffic for %s b/c there is no "
                         "conn in the db") proc-type conn-id)
            (rm-q conn-id))

          :else
          (do
            (send-q sender-or-excuse q conn-id proc-type)
            (rm-q conn-id))))))
  nil)

(s/defn enqueue :- nil
  [conn-id->waiting-q-atom conn-id bytes-seq msg-name conn-type conn-id
   proc-type]
  (debugf "%s enqueuing %s for %s %s" proc-type msg-name conn-type conn-id)
  (let [info (sym-map bytes-seq msg-name conn-type conn-id)]
    (if (contains? @conn-id->waiting-q-atom conn-id)
      (swap! conn-id->waiting-q-atom update conn-id conj info)
      (let [q (-> (u/make-queue)
                  (conj info))]
        (swap! conn-id->waiting-q-atom assoc conn-id q))))
  nil)

(defn- process-injection-command [command rcv-chan]
  (let [[op conn-id bytes] command]
    (ca/put! rcv-chan [conn-id bytes])))

(s/defn process-send-command :- nil
  [command db conn-id->waiting-q-atom proc-type]
  (let [[op conn-id & rest] command
        sender-or-excuse (get-sender-or-excuse db conn-id proc-type)
        [state conn-type] (state/get-conn-state-and-type db conn-id)
        [msg-name bytes-seq] (case op
                               :send-bytes ["bytes" rest]
                               :send-msg (let [[msg-name msg] rest]
                                           [msg-name (msg->enc-fragments
                                                      db msg-name msg)]))]
    (cond
      (= :not-ready sender-or-excuse)
      (enqueue conn-id->waiting-q-atom conn-id bytes-seq msg-name
               conn-type conn-id proc-type)

      ;; If there is no longer a row in the db (because the conn
      ;; timed out), just drop the traffic
      (= :not-present sender-or-excuse)
      (debugf "%s dropping %s for %s b/c there is no conn in the db"
              proc-type msg-name conn-id )

      :else
      (do-send sender-or-excuse bytes-seq msg-name conn-type
               conn-id proc-type))))

(defn- process-side-effect-command
  [command db-atom rcv-chan side-effect-op->f state-op->f
   conn-id->waiting-q-atom proc-type]
  (let [[op & rest] command
        f (side-effect-op->f op)
        pf (partial f @db-atom)
        command-or-block-seq (apply pf rest)]
    (doseq [command-or-block command-or-block-seq]
      (process-command-or-block
       command-or-block db-atom rcv-chan side-effect-op->f
       state-op->f conn-id->waiting-q-atom proc-type))))

(defn- process-state-command [command db-atom state-op->f proc-type]
  (let [[op & rest] command
        do-fn (fn [db]
                (let [f (-> (state-op->f op)
                            (partial db))]
                  (apply f rest)))]
    (swap! db-atom do-fn)))

(def inject-ops #{:inject-bytes})
(def send-ops #{:send-bytes :send-msg})
(def side-effect-op->f (sym-map close-and-delete-conn collect-garbage notify
                                prune-conns))

(defn- process-command-or-block
  [command-or-block db-atom rcv-chan side-effect-op->f state-op->f
   conn-id->waiting-q-atom proc-type]
  (let [commands (if (sequential? (first command-or-block))
                   command-or-block
                   [command-or-block])]
    (doseq [command commands]
      (let [[op & args] command]
        (when-not (contains? command-debug-blacklist op)
          (if (contains? command-debug-expanded op)
            (debugf "%s got %s" proc-type command)
            (debugf "%s got %s command" proc-type op)))
        (cond
          (contains? state-op->f op) (process-state-command
                                      command db-atom state-op->f proc-type)
          (contains? send-ops op) (process-send-command
                                   command @db-atom conn-id->waiting-q-atom
                                   proc-type)
          (contains? side-effect-op->f op) (process-side-effect-command
                                            command db-atom rcv-chan
                                            side-effect-op->f state-op->f
                                            conn-id->waiting-q-atom proc-type)
          (contains? inject-ops op) (process-injection-command
                                     command rcv-chan)
          :else (throw-far-error (str proc-type " got unknown command: " op)
                                 :execution-error :unknown-command
                                 (sym-map command)))))))

(defn- send-keep-alives [db proc-type]
  (let [conn-ids (state/get-logged-in-conn-ids db)]
    (doseq [conn-id conn-ids]
      (let [[sender conn-type] (state/get-sender-and-conn-type db conn-id)]
        (do-send sender [keep-alive-bytes] "keep-alive" conn-type conn-id
                 proc-type)))))

(defn- start-command-proc-loop
  [active?-atom db-atom command-out-chan rcv-chan side-effect-op->f state-op->f
   conn-id->waiting-q-atom proc-type]
  (u/go-sf
   (while @active?-atom
     (try
       (let [timeout-chan (ca/timeout mu/max-loop-wait-ms)
             [command-or-block-seq ch] (ca/alts!
                                        [command-out-chan timeout-chan])]
         (when (= command-out-chan ch)
           (doseq [command-or-block command-or-block-seq]
             (process-command-or-block
              command-or-block db-atom rcv-chan side-effect-op->f
              state-op->f conn-id->waiting-q-atom proc-type))))
       (catch #?(:clj Exception :cljs :default) e
         (errorf "Stopping processing due to exception: %s"
                 (u/get-exception-msg-and-stacktrace e))
         (reset! active?-atom false))))))

(s/defn start-processor :- nil
  [active?-atom db-atom repeater command-chan rcv-chan addl-side-effect-op->f
   addl-state-op->f addl-msg->op proc-type]
  (let [xf (map (fn [[conn-id bytes]]
                  (mxf/bytes->commands @db-atom conn-id bytes proc-type
                                       addl-msg->op)))
        xf-chan (ca/chan mu/chan-buf-size xf)
        _ (ca/pipe rcv-chan xf-chan)
        command-out-chan (ca/merge [command-chan xf-chan] mu/chan-buf-size)
        side-effect-op->f (merge side-effect-op->f addl-side-effect-op->f)
        state-op->f (merge state/op->f addl-state-op->f)
        conn-id->waiting-q-atom (atom {})]
    (pete/add-fn! repeater :keep-alive #(send-keep-alives @db-atom proc-type)
                  mu/keep-alive-interval-ms)
    (pete/add-fn! repeater :check-waiting-sends
                  #(check-waiting-sends @db-atom conn-id->waiting-q-atom
                                        proc-type)
                  mu/check-waiting-sends-interval-ms)
    (pete/add-fn! repeater :collect-garbage
                  #(ca/put! command-chan [[:collect-garbage]])
                  mu/gc-interval-ms)
    (start-command-proc-loop
     active?-atom db-atom command-out-chan rcv-chan side-effect-op->f
     state-op->f conn-id->waiting-q-atom proc-type)
    nil))
