(ns farbetter.mu.state
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [#?(:clj clojure.core.async.impl.protocols
       :cljs cljs.core.async.impl.protocols) :refer [Channel]]
   [clojure.string :refer [join]]
   [farbetter.mu.msgs :as msgs]
   [farbetter.mu.utils :as mu :refer
    [max-fragment-bytes Command ConnId ConnState Fingerprint Fragment MsgId
     OpToF ProcType RequestId]]
   [farbetter.roe :as roe]
   [farbetter.roe.schemas :as rs :refer [AvroData AvroName AvroSchema]]
   [farbetter.utils :as u :refer
    [throw-far-error ByteArray #?@(:clj [go-safe inspect sym-map])]]
   [freedomdb.frontend :as fdb]
   [freedomdb.schemas :refer [DBType]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [go-safe inspect sym-map]])))

(declare add-edn-schema send-msg)

(def keep-alive-msg {:keep-alive true})
(def keep-alive-interval-ms (* 1000 20))
(def connect-timeout-ms (* 1000 10))
(def schema-rq-interval-ms (* 1000 5))
(def max-fragment-wait-ms keep-alive-interval-ms)
(def max-schema-rq-wait-ms keep-alive-interval-ms)

(defn- add-api-schema [db schema]
  (let [{:keys [name]} schema
        fingerprint (roe/edn-schema->fingerprint schema)]
    (if (fdb/select db {:tables [:schemas]
                        :where [:= :fingerprint fingerprint]})
      db
      (fdb/insert db :schemas (sym-map name fingerprint schema)))))

(defn add-api [db api]
  (let [{:keys [service-name major-version fn-schemas]} api
        add-row (fn [db [fn-name {:keys [arg-schema return-schema]}]]
                  (let [row (sym-map service-name major-version fn-name
                                     arg-schema return-schema)
                        arg-fp (roe/edn-schema->fingerprint arg-schema)
                        ret-fp (roe/edn-schema->fingerprint return-schema)]
                    (-> db
                        (fdb/insert :apis row)
                        (add-edn-schema arg-fp arg-schema)
                        (add-edn-schema ret-fp return-schema))))]
    (reduce add-row db fn-schemas)))

(s/defn populate-schemas-table :- DBType
  [db :- DBType]
  (reduce add-api-schema db mu/msg-schemas))

(defn populate-api-table [db apis]
  (reduce add-api db apis))

(s/defn make-db :- DBType
  ([apis :- [msgs/APISpec]]
   (make-db apis identity))
  ([apis :- [msgs/APISpec]
    modifier :- (s/=> DBType DBType)]
   (-> (fdb/create-db :mem)
       (fdb/create-table
        :conns {:conn-id {:type :any :indexed true}
                :sender {:type :any :indexed false}
                :closer {:type :any :indexed false}
                :type {:type :kw :indexed true}
                :state {:type :kw :indexed true}
                :start-time-ms {:type :int8 :indexed true}})
       (fdb/create-table
        :fragments {:msg-id {:type :any :indexed true} ;; msg-id is a map
                    :fragment-num {:type :int4 :indexed true}
                    :fragment-data {:type :any :indexed false}
                    :insert-time-ms {:type :int8 :indexed true}})
       (fdb/create-table
        :schemas {:name {:type :any :indexed true} ;; can be nil
                  :fingerprint {:type :any :indexed true}
                  :schema {:type :any :indexed true}})
       (fdb/create-table
        :schema-rqs {:fingerprint {:type :any :indexed true}
                     :insert-time-ms {:type :int8 :indexed true}})
       (fdb/create-table
        :bytes-waiting-for-schemas
        {:fingerprint {:type :any :indexed true}
         :conn-id {:type :any :indexed true}
         :bytes {:type :any :indexed false}
         :insert-time-ms {:type :int8 :indexed true}})
       (fdb/create-table
        :apis {:service-name {:type :str1000 :indexed true}
               :major-version {:type :int4 :indexed true}
               :fn-name {:type :str1000 :indexed true}
               :arg-schema {:type :any :indexed false}
               :return-schema {:type :any :indexed false}})
       (fdb/create-table
        :to-send {:id {:type :any :indexed true}
                  :conn-id {:type :any :indexed false}
                  :bytes {:type :any :indexed false}})
       (populate-schemas-table)
       (populate-api-table apis)
       (modifier))))

(s/defn get-waiting-bytes :- [[(s/one ConnId "conn-id")
                               (s/one ByteArray "bytes")]]
  [db :- DBType
   fingerprint :- Fingerprint]
  (fdb/select db {:tables [:bytes-waiting-for-schemas]
                  :fields [:conn-id :bytes]
                  :where [:= :fingerprint fingerprint]}))

(s/defn encode-fragment :- ByteArray
  [fragment :- Fragment]
  (roe/edn->avro-byte-array msgs/fragment-schema fragment))

(defn make-fragment [msg-id fingerprint data fragment-num num-fragments
                     last-fragment-size]
  (let [chunk-start (* max-fragment-bytes fragment-num)
        chunk-len (if (and (= (dec num-fragments)
                              fragment-num)
                           (not (zero? last-fragment-size)))
                    last-fragment-size
                    max-fragment-bytes)
        chunk-end (+ chunk-start chunk-len)
        fragment-data (u/slice data chunk-start chunk-end)]
    (sym-map msg-id num-fragments fragment-num fingerprint fragment-data)))

(s/defn data->fragments* :- [(s/if u/byte-array? ByteArray Fragment)]
  [fingerprint :- Fingerprint
   data :- ByteArray
   encode? :- s/Bool]
  (let [data-len (count data)
        num-full-fragments (quot data-len max-fragment-bytes)
        last-fragment-size (rem data-len max-fragment-bytes)
        num-fragments (if (zero? last-fragment-size)
                        num-full-fragments
                        (inc num-full-fragments))
        msg-id (when (> num-fragments 1)
                 (-> (u/make-v1-uuid)
                     (u/uuid->int-map)))
        mf (fn [acc fragment-num]
             (let [fragment (make-fragment
                             msg-id fingerprint data fragment-num
                             num-fragments last-fragment-size)]
               (conj acc (if encode?
                           (encode-fragment fragment)
                           fragment))))]
    (reduce mf [] (range num-fragments))))

(s/defn data->fragments :- [Fragment]
  [fingerprint :- Fingerprint
   data :- ByteArray]
  (data->fragments* fingerprint data false))

(s/defn data->enc-fragments :- [ByteArray]
  [fingerprint :- Fingerprint
   data :- ByteArray]
  (data->fragments* fingerprint data true))

(s/defn add-fragment :- DBType
  [db :- DBType
   msg-id :- MsgId
   fragment-num :- s/Num
   fragment-data :- ByteArray]
  (let [insert-time-ms (u/long (u/get-current-time-ms))]
    (if (fdb/select db {:tables [:fragments]
                        :where [:and
                                [:= :msg-id msg-id]
                                [:= :fragment-num fragment-num]]})
      db ;; fragment exists
      (fdb/insert db :fragments (sym-map msg-id fragment-num fragment-data
                                         insert-time-ms)))))

(s/defn get-fragment-datas :- [ByteArray]
  [db :- DBType
   msg-id :- MsgId]
  (fdb/select db {:tables [:fragments]
                  :where [:= :msg-id msg-id]
                  :fields :fragment-data
                  :order-by [:fragment-num :asc]}))

(s/defn fingerprint->schema :- (s/maybe AvroSchema)
  [db :- DBType
   fingerprint :- Fingerprint]
  (fdb/select-one db {:tables [:schemas]
                      :fields :schema
                      :where [:= :fingerprint fingerprint]}))

(s/defn schema->fingerprint :- (s/maybe Fingerprint)
  [db :- DBType
   schema :- AvroSchema]
  (fdb/select-one db {:tables [:schemas]
                      :fields :fingerprint
                      :where [:= :schema schema]}))

(s/defn name->schema :- (s/maybe AvroSchema)
  [db :- DBType
   name :- AvroName]
  (fdb/select-one db {:tables [:schemas]
                      :fields :schema
                      :where [:= :name name]}))

(s/defn fn-name->schemas :- {(s/required-key :arg-schema) AvroData
                             (s/required-key :return-schema) AvroData}
  ([db :- DBType
    fn-name :- s/Str]
   (fn-name->schemas db nil fn-name))
  ([db :- DBType
    service-name :- (s/maybe s/Str)
    fn-name :- s/Str]
   (let [where (cond-> [:and
                        [:= :fn-name fn-name]]
                 service-name (conj [:= :service-name service-name]))
         row (fdb/select-one db {:tables [:apis]
                                 :where where})]
     (select-keys row [:arg-schema :return-schema]))))

(s/defn get-conn-state :- (s/maybe ConnState)
  [db :- DBType
   conn-id :- ConnId]
  (fdb/select-one db {:tables [:conns]
                      :where [:= :conn-id conn-id]
                      :fields :state}))

(s/defn set-sender-closer! :- nil
  [db-atom :- (s/atom DBType)
   conn-id :- ConnId
   sender :- (s/=> s/Any)
   closer :- (s/=> s/Any)]
  (swap! db-atom fdb/update :conns {:set (sym-map sender closer)
                                    :where [:= :conn-id conn-id]}))

(s/defn set-conn-state :- DBType
  [db :- DBType
   conn-id :- ConnId
   state :- ConnState]
  (when-not (mu/valid-conn-states state)
    (throw-far-error (str "Invalid connection state `" state "`")
                     :illegal-argument :invalid-connection-state
                     (sym-map state)))
  (fdb/update db :conns {:set (sym-map state)
                         :where [:= :conn-id conn-id]}))

(s/defn get-conn-ids :- [s/Str]
  [db :- DBType]
  (fdb/select db {:tables [:conns]
                  :fields :conn-id}))

(defn- make-fn-schema [service-name fn-name schema-type schema]
  (let [name (join "." [service-name fn-name schema-type schema])]
    (if (map? schema)
      (assoc schema :name name)
      {:name name
       :type schema})))

(s/defn timed-out? :- s/Bool
  [db :- DBType
   conn-id :- ConnId]
  (let [start-time-ms (fdb/select-one db {:tables [:conns]
                                          :fields :start-time-ms
                                          :where [:= :conn-id conn-id]})]
    (#?(:clj < :cljs .lt)
     (#?(:clj + :cljs .add) start-time-ms connect-timeout-ms)
     (u/long (u/get-current-time-ms)))))

(s/defn get-logged-in-conn-ids :- [ConnId]
  [db :- DBType]
  (fdb/select db {:tables [:conns]
                  :fields :conn-id
                  :where [:= :state :logged-in]
                  :order-by [:conn-id :asc]}))

(s/defn get-gw-conn-id :- (s/maybe ConnId)
  [db :- DBType]
  (first (get-logged-in-conn-ids db)))

(s/defn get-non-logged-in-conn-ids :- [ConnId]
  [db :- DBType]
  (fdb/select db {:tables [:conns]
                  :fields :conn-id
                  :where [:not [:= :state :logged-in]]
                  :order-by [:conn-id :asc]}))

(s/defn add-conn! :- nil
  [db-atom :- (s/atom DBType)
   conn-id :- ConnId
   sender :- (s/maybe (s/=> s/Any))
   closer :- (s/maybe (s/=> s/Any))
   type :- ProcType]
  (debugf "Entering add-conn! conn-id: %s" conn-id)
  (let [state :start
        start-time-ms (u/long (u/get-current-time-ms))
        row (sym-map conn-id sender closer type state start-time-ms)]
    (swap! db-atom fdb/insert :conns row)
    nil))

(s/defn remove-conn! :- nil
  [db-atom :- (s/atom DBType)
   conn-id :- ConnId]
  (swap! db-atom fdb/delete :conns [:= :conn-id conn-id])
  nil)

(s/defn close-conn :- DBType
  [db :- DBType
   conn-id :- ConnId]
  (if-let [closer (fdb/select-one db {:tables [:conns]
                                      :fields :closer
                                      :where [:= :conn-id conn-id]})]
    (try
      (debugf "Calling closer for %s" conn-id)
      (closer)
      db
      (catch #?(:clj Exception :cljs :default) e
        (errorf "Closer threw an exception: %s" e)
        db))
    db))

(s/defn close-conns :- DBType
  [db :- DBType
   conn-ids :- [ConnId]]
  (reduce close-conn db conn-ids))

(s/defn close-all-conns :- DBType
  [db :- DBType]
  (let [conn-ids (fdb/select db {:tables [:conns]
                                 :fields :conn-id})]
    (reduce close-conn db conn-ids)))

(s/defn prune-conns :- DBType
  "Remove all :logged-in connections except one."
  [db :- DBType]
  (let [li-conn-ids (get-logged-in-conn-ids db)]
    (debugf "In prune-conns. li-conn-ids: %s" li-conn-ids)
    (reduce close-conn db (rest li-conn-ids))))

(s/defn handle-missing-schema :- DBType
  [db :- DBType
   conn-id :- ConnId
   fingerprint :- Fingerprint
   bytes :- ByteArray]
  (let [rq-time-ms (fdb/select db {:tables [:schema-rqs]
                                   :fields :insert-time-ms
                                   :where [:= :fingerprint fingerprint]})
        rq-time? (or (not rq-time-ms)
                     (#?(:clj > :cljs .gt) (u/get-current-time-ms)
                        (#?(:clj + :cljs .add) rq-time-ms
                           schema-rq-interval-ms)))
        insert-time-ms (u/long (u/get-current-time-ms))]
    (cond-> db
      rq-time? (send-msg conn-id :schema-rq (sym-map fingerprint))
      true (fdb/insert :bytes-waiting-for-schemas
                       (sym-map fingerprint bytes conn-id insert-time-ms)))))

(s/defn process-msg :- DBType
  "This command should be overridden by the processor."
  [db :- DBType
   conn-id :- ConnId
   msg-name :- AvroName
   msg :- AvroData]
  (throw-far-error "process-msg was not overridden."
                   :execution-error :process-msg-not-overridden
                   (sym-map msg-name)))

(s/defn remove-fragments :- DBType
  [db :- DBType
   msg-id :- MsgId]
  (fdb/delete db :fragments [:= :msg-id msg-id]))

(s/defn send-bytes :- DBType
  [db :- DBType
   conn-id :- ConnId
   bytes :- ByteArray]
  (let [id (u/make-v1-uuid)]
    (fdb/insert db :to-send (sym-map id conn-id bytes))))

(s/defn send-msg :- DBType
  [db :- DBType
   conn-id :- ConnId
   msg-name :- AvroName
   msg :- AvroData]
  (let [schema (name->schema db msg-name)
        _ (when-not schema
            (throw-far-error
             (str "Unknown msg-name: " msg-name)
             :execution-error :unknown-msg-name
             (sym-map msg-name conn-id msg)))
        fingerprint (schema->fingerprint db schema)
        data (roe/edn->avro-byte-array schema msg)
        _ (mu/check-data-len (count data))
        enc-fragments (data->enc-fragments fingerprint data)]
    (reduce (fn [db fragment]
              (send-bytes db conn-id fragment))
            db enc-fragments)))

(s/defn send-schema :- DBType
  [db :- DBType
   conn-id :- ConnId
   fingerprint :- Fingerprint]
  (let [edn-schema (fingerprint->schema db fingerprint)
        json-schema (roe/edn-schema->json-schema edn-schema)]
    (send-msg db conn-id :fingerprint-to-json-schema
              (sym-map fingerprint json-schema))))



(s/defn add-edn-schema :- DBType
  [db :- DBType
   fingerprint :- Fingerprint
   edn-schema :- AvroSchema]
  (let [{:keys [name]} edn-schema
        schema edn-schema
        existing-row (fdb/select db {:tables [:schemas]
                                     :where [:= :fingerprint fingerprint]})]
    (if-not existing-row
      (fdb/insert db :schemas (sym-map name fingerprint schema))
      db)))

(s/defn add-schema :- DBType
  [db :- DBType
   fingerprint :- Fingerprint
   json-schema :- s/Str]
  (let [edn-schema (roe/json-schema->edn-schema json-schema)]
    (add-edn-schema db fingerprint edn-schema)))

(s/defn remove-waiting-bytes :- DBType
  [db :- DBType
   fingerprint :- Fingerprint]
  (fdb/delete db :bytes-waiting-for-schemas [:= :fingerprint fingerprint]))

(def op->f
  (sym-map add-fragment add-schema close-conn handle-missing-schema process-msg
           remove-fragments remove-waiting-bytes send-bytes send-msg send-schema
           set-conn-state))

(s/defn process-command :- DBType
  [db :- DBType
   command :- Command
   addl-op->f :- OpToF]
  ;; TODO: pre-merge op->f and addl-op->f
  (let [op->f (merge op->f addl-op->f)
        [op & rest] command
        f (-> (op->f op)
              (partial db))]
    (apply f rest)))

(defn get-threshold-ms [max-wait-ms]
  (#?(:clj - :cljs .sub) (u/long (u/get-current-time-ms))
     max-wait-ms))

(s/defn gc-fragments :- DBType
  [db :- DBType]
  (let [max-time-ms (get-threshold-ms max-fragment-wait-ms)]
    (fdb/delete db :fragments [:< :insert-time-ms max-time-ms])))

(s/defn gc-schema-rqs :- DBType
  [db :- DBType]
  (let [max-time-ms (get-threshold-ms max-fragment-wait-ms)]
    (fdb/delete db :schema-rqs [:< :insert-time-ms max-time-ms])))

(s/defn gc-msgs-waiting :- DBType
  [db :- DBType]
  (let [max-time-ms (get-threshold-ms max-fragment-wait-ms)]
    (fdb/delete db :bytes-waiting-for-schemas [:< :insert-time-ms
                                               max-time-ms])))
