(ns farbetter.mu.gateway
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.mu.gw-router :as router]
   [farbetter.mu.msgs :as msgs]
   [farbetter.mu.proc :as proc]
   [farbetter.mu.state :as state]
   [farbetter.mu.utils :as mu :refer [ConnId MsgId]]
   [farbetter.pete :as pete]
   [farbetter.roe :as roe]
   [farbetter.roe.schemas :as rs :refer [AvroData AvroName]]
   [farbetter.utils :as u :refer
    [throw-far-error ByteArray #?@(:clj [go-safe inspect sym-map])]]
   [freedomdb.frontend :as fdb]
   [freedomdb.schemas :refer [DBType]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof warnf]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [go-safe inspect sym-map]])))

(declare make-on-connect make-op->f modify-db gc refresh-rules)

(def rcv-chan-buf-size 1000)
(def rules-refresh-interval-ms (* 1000 2))
(def max-rpc-wait-ms (* 1000 20))

(defprotocol IGateway
  (serve [this])
  (stop [this]))

(defrecord Gateway [active?-atom db-atom repeater serve-fn stop-fn-atom
                    rcv-chan]
  IGateway
  (serve [this]
    (let [client-ws-handler (make-on-connect :cl db-atom rcv-chan)
          si-ws-handler (make-on-connect :si db-atom rcv-chan)]
      (reset! stop-fn-atom (serve-fn client-ws-handler si-ws-handler))))
  (stop [this]
    (reset! active?-atom false)
    (pete/stop repeater)
    (@stop-fn-atom)))

(s/defn make-gateway :- (s/protocol IGateway)
  [user-validator :- (s/=> s/Any)
   si-validator :- (s/=> s/Any)
   rules-factory :- (s/=> s/Any)
   serve-fn :- (s/=> s/Any)]
  (let [db-atom (atom (state/make-db [] modify-db))
        repeater (pete/make-repeater)
        active?-atom (atom true)
        rcv-chan (ca/chan rcv-chan-buf-size)
        stop-fn-atom (atom (constantly nil))
        op->f (make-op->f user-validator si-validator)]
    ;; Explicitly call this the first time to ensure that rules are
    ;; populated for testing
    (refresh-rules db-atom rules-factory)
    (pete/add-fn repeater :refresh-rules
                 #(refresh-rules db-atom rules-factory)
                 rules-refresh-interval-ms)
    (proc/start-procs repeater active?-atom db-atom rcv-chan :gw op->f gc)
    (proc/start-data-processing-loop active?-atom db-atom rcv-chan :gw op->f)
    (map->Gateway
     (sym-map active?-atom db-atom repeater serve-fn stop-fn-atom rcv-chan))))


;;;;;;;;;;;;;;;;;;;; Helper fns ;;;;;;;;;;;;;;;;;;;;

(defn make-on-connect [conn-type db-atom rcv-chan]
  (fn on-connect [conn-id sender closer]
    (let [on-disconnect #(do
                           (debugf "GW on-disconnect. Reason: %s" %)
                           (state/remove-conn! db-atom conn-id))
          on-error #(do (debugf "Error: %s" %)
                        (on-disconnect %))
          on-rcv (fn [data]
                   (ca/put! rcv-chan [conn-id data]))]
      (state/add-conn! db-atom conn-id sender closer conn-type)
      (swap! db-atom state/set-conn-state conn-id :connected)
      (sym-map on-rcv on-error on-disconnect))))

(defn- handle-rpc-rq [db client-conn-id msg]
  (let [{:keys [request-id arg-schema-fp]} msg]
    (if (state/fingerprint->schema db arg-schema-fp)
      (let [insert-time-ms (u/long (u/get-current-time-ms))
            si-conn-id (router/route-rpc-rq db msg)]
        (debugf "si-conn-id: %s" si-conn-id)
        (if si-conn-id
          (-> db
              (state/send-msg si-conn-id :rpc-rq msg)
              (fdb/insert :gw-rpcs (sym-map client-conn-id si-conn-id request-id
                                            insert-time-ms)))
          (let [request-id-uuid (u/int-map->uuid request-id)
                request-id-str (u/uuid->hex-str request-id-uuid)
                reason (str "No matching traffic policy rule or service "
                            "instance found for rpc-request. Request-id:"
                            request-id-str)
                rq-failed-msg (sym-map request-id reason)]
            (warnf reason)
            (state/send-msg db client-conn-id :rq-failed rq-failed-msg))))
      (let [fingerprint (roe/edn-schema->fingerprint msgs/rpc-rq-schema)
            fragment-data (roe/edn->avro-byte-array msgs/rpc-rq-schema msg)
            fragment (sym-map fingerprint fragment-data)
            bytes (roe/edn->avro-byte-array msgs/fragment-schema fragment)]
        (state/handle-missing-schema db client-conn-id arg-schema-fp bytes)))))

(defn- handle-rpc-rs [db si-conn-id msg]
  (let [{:keys [request-id return-schema-fp]} msg]
    (if (state/fingerprint->schema db return-schema-fp)
      (let [client-conn-id (router/request-id->client-conn-id db request-id)]
        (cond-> db
          client-conn-id (state/send-msg client-conn-id :rpc-rs msg)
          true (fdb/delete :gw-rpcs [:= :request-id request-id])))
      (let [fingerprint (roe/edn-schema->fingerprint msgs/rpc-rs-schema)
            fragment-data (roe/edn->avro-byte-array msgs/rpc-rs-schema msg)
            fragment (sym-map fingerprint fragment-data)
            bytes (roe/edn->avro-byte-array msgs/fragment-schema fragment)]
        (state/handle-missing-schema db si-conn-id return-schema-fp bytes)))))

(defn- handle-rq-failed [db si-conn-id msg]
  (let [{:keys [request-id]} msg
        client-conn-id (router/request-id->client-conn-id db request-id)]
    (cond-> db
      client-conn-id (state/send-msg client-conn-id :rq-failed msg)
      true (fdb/delete :gw-rpcs [:= :request-id request-id]))))

(defn make-si-login-handler [si-validator]
  (fn [db conn-id msg]
    (let [{:keys [access-key service-instance-version]} msg
          was-successful (si-validator access-key)]
      (cond-> db
        was-successful (fdb/insert :sis (assoc service-instance-version
                                               :si-conn-id conn-id))
        true (state/send-msg conn-id :service-instance-login-rs
                             (sym-map was-successful))))))

(defn make-op->f [user-validator si-validator]
  (let [handle-client-login (fn [db conn-id msg]
                              (let [{:keys [email password]} msg
                                    id (user-validator email password)
                                    rs-msg {:was-successful (boolean id)
                                            :user-id id}]
                                (-> db
                                    (state/set-conn-state conn-id :logged-in)
                                    (state/send-msg conn-id :client-login-rs
                                                    rs-msg))))
        handle-si-login (make-si-login-handler si-validator)
        process-msg (fn [db conn-id msg-name msg]
                      (let [f (case msg-name
                                :client-login-rq handle-client-login
                                :service-instance-login-rq handle-si-login
                                :rpc-rq handle-rpc-rq
                                :rpc-rs handle-rpc-rs
                                :rq-failed handle-rq-failed)]
                        (f db conn-id msg)))]
    (sym-map process-msg)))

(defn- modify-db [db]
  [db :- DBType]
  (-> db
      (fdb/create-table :gw-rpcs {:client-conn-id {:type :any :indexed true}
                                  :si-conn-id {:type :any :indexed true}
                                  :request-id {:type :any :indexed true}
                                  :insert-time-ms {:type :int8 :indexed true}})
      (fdb/create-table :sis {:si-conn-id {:type :any}
                              :service-name {:type :str1000}
                              :major-version {:type :int4}
                              :minor-version {:type :int4}
                              :micro-version {:type :int4}})
      (fdb/create-table :traffic-policy-rules
                        {:service-name {:type :str1000}
                         :major-version {:type :int4}
                         :minor-version {:type :int4}
                         :micro-version {:type :int4}
                         :users {:type :any}
                         :weight {:type :int4}})))

(s/defn gc :- DBType
  [db :- DBType]
  (let [max-time-ms (#?(:clj - :cljs .sub) (u/long (u/get-current-time-ms))
                       max-rpc-wait-ms)]
    (fdb/delete db :gw-rpcs [:< :insert-time-ms max-time-ms])))

(defn- refresh-rules [db-atom rules-factory]
  (let [add-row (fn [db row]
                  (fdb/insert db :traffic-policy-rules row))
        add-rows (fn [db]
                   (reduce add-row db (rules-factory)))
        refresh (fn [db]
                  (-> db
                      (fdb/delete :traffic-policy-rules)
                      (add-rows)))]
    (swap! db-atom refresh)))
