(ns farbetter.mu.service-instance
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [#?(:clj clojure.core.async.impl.protocols
       :cljs cljs.core.async.impl.protocols) :refer [Channel]]
   [farbetter.mu.msgs :as msgs :refer [APISpec FnMap]]
   [farbetter.mu.msg-xf :as mx]
   [farbetter.mu.proc :as proc]
   [farbetter.mu.state :as state]
   [farbetter.mu.transport :as tp]
   [farbetter.mu.utils :as mu :refer [Command ConnState ProcType]]
   [farbetter.mu.websocket :as websocket]
   [farbetter.pete :as pete]
   [farbetter.roe :as roe]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [go-safe inspect sym-map])]]
   [freedomdb.schemas :refer [DBType]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [go-safe inspect sym-map]])))

(declare map->ServiceInstance make-msg-processor)

(def rcv-chan-buf-size 100)
(def gw-conn-update-interval-ms (* 1000 2))

;;;;;;;;;;;;;;;;;;;; Protocol & Record ;;;;;;;;;;

(defprotocol IServiceInstance
  (start [this] "Start serving requests.")
  (stop [this] "Stop serving requests.")
  (update-conns [this] "Update gateway connections."))

(s/defn make-service-instance :- (s/protocol IServiceInstance)
  "Create a service instance.
   Arguments:
   - api - Map describing the API of the service instance. Required keys:
        - :service-name - String. Name of the service.
        - :major-version - Integer. Major version number of the API.
        - :fns - Map of fn-name keys to fns of one arg
   - svc-fns - Map of fn-name keys to fn values.
   - minor-version - Integer. Minor version of this implementation.
   - micro-version - Integer. Micro version of this implementation.
   - access-key - String. Service instance access key. Used for GW login.
   - gw-urls-factory - Fn of no arguments. Should return a :success/:failure
        channel. If successful, the second argument should be a sequence of
        valid gateway URLs (as strings). Optional.
   - tp-factory - Factory fn to create a transport layer. Optional. Must be a
        fn of five arguments:
        - url - String. URL of the particular gateway.
        - on-rcv - Fn of one argument (data). Transport layer should call
             this fn when it receives data.
        - on-connect - Fn of no arguments. Transport layer should call this
             fn when it connects or reconnects.
        - on-disconnect - Fn of no arguments. Transport layer should call
             this fn when it disconnections.
        - on-error - Fn of one argument (error). Transport layer should call
             this fn when it encounters an error.
        Calling tp-factory must return a sender fn of one argument (data).
             This fn will be called to send data to the gateway."
  [api :- APISpec
   svc-fns :- FnMap
   minor-version :- s/Int
   micro-version :- s/Int
   access-key :- s/Str
   gw-urls-factory :- (s/=> [s/Str])
   tp-factory :- (s/=> s/Any)]
  (let [{:keys [service-name major-version fns]} api
        api-fn-names (set (keys fns))
        svc-fn-names (set (keys svc-fns))
        missing-fns (clojure.set/difference api-fn-names svc-fn-names)
        _ (when (seq missing-fns)
            (throw-far-error (str "Missing svc-fn(s):" missing-fns)
                             :illegal-argument :missing-fns
                             (sym-map api svc-fns missing-fns)))
        repeater (pete/make-repeater)
        service-instance-version (sym-map service-name major-version
                                          minor-version micro-version)
        rcv-chan (ca/chan rcv-chan-buf-size)
        db-atom (atom (state/make-db [api]))
        active?-atom (atom true)
        params (sym-map db-atom gw-urls-factory tp-factory repeater rcv-chan
                        access-key svc-fns api service-instance-version
                        active?-atom)
        si (map->ServiceInstance params)
        send-login (fn [db conn-id]
                     (let [msg (sym-map access-key service-instance-version)]
                       (state/send-msg db conn-id
                                       :service-instance-login-rq msg)))
        process-msg (make-msg-processor si)

        addl-op->f (sym-map send-login process-msg)]
    (pete/add-fn repeater :update-conns #(update-conns si)
                 gw-conn-update-interval-ms)
    (proc/start-procs repeater active?-atom db-atom rcv-chan :si addl-op->f
                      identity)
    (start si)
    si))

(defrecord ServiceInstance [db-atom gw-urls-factory tp-factory repeater rcv-chan
                            access-key svc-fns api service-instance-version
                            active?-atom]
  IServiceInstance
  (start [this]
    (reset! active?-atom true)
    (pete/start repeater))

  (stop [this]
    (reset! active?-atom false)
    (pete/stop repeater))

  (update-conns [this]
    (go-safe
     (let [[status arg] (ca/<! (gw-urls-factory))
           _ (when-not (= :success status)
               (throw-far-error
                (str "gw-urls-factory call failed. reason: " arg)
                :execution-error :gw-urls-call-failed {:reason arg}))
           gw-urls (set arg)
           current-urls (set (state/get-conn-ids @db-atom))
           added-urls (clojure.set/difference gw-urls current-urls)
           deleted-urls (clojure.set/difference current-urls gw-urls)]
       (doseq [url added-urls]
         (tp/connect-to-gw db-atom url rcv-chan tp-factory))
       (swap! db-atom state/close-conns (seq deleted-urls))))))

;;;;;;;;;;;;;;;;;;;; Helper fns ;;;;;;;;;;;;;;;;;;;;

(defn handle-login-rs [si db conn-id msg]
  (if (:was-successful msg)
    (state/set-conn-state db conn-id :logged-in)
    (throw-far-error "SI login failed due to incorrect access key."
                     :execution-error :failed-si-login {})))

(defn execute-fn [db-atom conn-id request-id return-schema fn-name f arg]
  (go-safe
   (try
     (let [ret (f arg)
           return-schema-fp (roe/edn-schema->fingerprint return-schema)
           encoded-return-value (roe/edn->avro-byte-array return-schema ret)
           ret-msg (sym-map request-id return-schema-fp encoded-return-value)]
       (swap! db-atom state/send-msg conn-id :rpc-rs ret-msg))
     (catch #?(:clj Exception :cljs :default) e
       (let [reason (str "Exception in service-fn `" fn-name "` "
                         "request-id: " request-id " Exception: " e)]
         (infof reason)
         (swap! db-atom state/send-msg conn-id :rq-failed
                (sym-map request-id reason)))))))

(defn handle-rpc-rq [si db conn-id msg db-atom]
  (let [{:keys [request-id fn-name arg-schema-fp encoded-arg]} msg
        w-schema (state/fingerprint->schema db arg-schema-fp)]
    (if w-schema
      (let [{:keys [arg-schema return-schema]} (state/fn-name->schemas
                                                db fn-name)
            arg (roe/avro-byte-array->edn w-schema arg-schema encoded-arg)
            f (get-in si [:svc-fns fn-name])]
        (execute-fn db-atom conn-id request-id return-schema fn-name f arg)
        db)
      (let [fingerprint (roe/edn-schema->fingerprint msgs/rpc-rq-schema)
            fragment-data (roe/edn->avro-byte-array msgs/rpc-rq-schema msg)
            fragment (sym-map fingerprint fragment-data)
            bytes (roe/edn->avro-byte-array msgs/fragment-schema fragment)]
        (state/handle-missing-schema db conn-id arg-schema-fp bytes)))))

(defn make-msg-processor [si]
  (fn process-msg [db conn-id msg-name msg]
    (case msg-name
      :service-instance-login-rs (handle-login-rs si db conn-id msg)
      :rpc-rq (handle-rpc-rq si db conn-id msg (:db-atom si))
      (throw-far-error
       "Got unknown msg type."
       :execution-error :rcvd-unknown-msg-type
       (sym-map msg-name msg conn-id)))))
