(ns farbetter.mu.msg-xf
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [#?(:clj clojure.core.async.impl.protocols
       :cljs cljs.core.async.impl.protocols) :refer [Channel]]
   [farbetter.mu.msgs :as msgs]
   [farbetter.mu.state :as state]
   [farbetter.mu.utils :as mu :refer
    [Command ConnId ConnState ProcType]]
   [farbetter.roe :as roe]
   [farbetter.roe.schemas :as rs]
   [farbetter.utils :as u :refer
    [throw-far-error ByteArray #?@(:clj [go-safe inspect sym-map])]]
   [freedomdb.frontend :as fdb]
   [freedomdb.schemas :refer [DBType]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [go-safe inspect sym-map]])))

;;;;;;;;;;;;;;;;;;;; Public Fns ;;;;;;;;;;;;;;;;;;;;

(defn- handle-schema-rs [db conn-id msg]
  (let [{:keys [fingerprint json-schema]} msg
        rows (state/get-waiting-bytes db fingerprint)
        init-commands [[:add-schema fingerprint json-schema]]
        commands (reduce (fn [acc row]
                           (let [[conn-id bytes] row]
                             (conj acc [:inject-bytes conn-id bytes])))
                         init-commands rows)]
    (conj commands [:remove-waiting-bytes fingerprint])))

(defn- msg->commands [db conn-id msg-name msg proc-type]
  (let [state (state/get-conn-state db conn-id)]
    (debugf "%s got %s msg" proc-type msg-name)
    (case msg-name
      :keep-alive [] ;; do nothing on keepalive
      :schema-rq [[:send-schema conn-id (:fingerprint msg)]]
      :fingerprint-to-json-schema (handle-schema-rs db conn-id msg)
      [[:process-msg conn-id msg-name msg]])))


(defn- enc-msg->commands [db conn-id fingerprint enc-msg proc-type]
  (if-let [writer-schema (state/fingerprint->schema db fingerprint)]
    (let [msg-name (:name writer-schema)
          reader-schema (state/name->schema db msg-name)
          msg (roe/avro-byte-array->edn writer-schema reader-schema enc-msg)]
      (msg->commands db conn-id msg-name msg proc-type))
    (let [retry-fragment {:msg-id nil
                          :num-fragments 1
                          :fragment-num 0
                          :fingerprint fingerprint
                          :fragment-data enc-msg}
          bytes (roe/edn->avro-byte-array msgs/fragment-schema retry-fragment)]
      [[:handle-missing-schema conn-id fingerprint bytes]])))

(defn- bytes->commands* [db conn-id bytes proc-type]
  (let [fragment (roe/avro-byte-array->edn msgs/fragment-schema
                                           msgs/fragment-schema bytes)
        {:keys [msg-id num-fragments fragment-num
                fingerprint fragment-data]} fragment]
    (if (= 1 num-fragments)
      (enc-msg->commands db conn-id fingerprint fragment-data proc-type)
      (loop [db db]
        (let [fragment-datas (state/get-fragment-datas db msg-id)
              num-fragment-datas (count fragment-datas)
              af-command [:add-fragment msg-id fragment-num fragment-data]]
          (if (= (dec num-fragments) num-fragment-datas)
            ;; Last frag, but not yet in db
            (recur (state/process-command db af-command {}))
            ;; Last frag and it's in the db
            (if (= num-fragments num-fragment-datas)
              (let [msg-data (u/concat-byte-arrays fragment-datas)]
                (mu/check-data-len (count msg-data))
                (conj (enc-msg->commands db conn-id fingerprint
                                         msg-data proc-type)
                      [:remove-fragments msg-id]))
              ;; otherwise, just add the fragment
              [af-command])))))))

(defn- bytes->commands-connected [db conn-id bytes proc-type]
  (let [peer-protocol-version (roe/avro-byte-array->edn :int :int bytes)]
    (if (= :gw proc-type)
      (if (= mu/protocol-version peer-protocol-version)
        [[:send-bytes conn-id bytes]
         [:set-conn-state conn-id :protocol-negotiated]]
        [[:send-bytes conn-id (roe/edn->avro-byte-array :int 0)]
         [:close-conn conn-id]])
      (if (zero? peer-protocol-version)
        [[:close-conn conn-id]]
        [[:set-conn-state conn-id :protocol-negotiated]
         [:send-login conn-id]]))))

(s/defn bytes->commands :- [Command]
  [db :- DBType
   conn-id :- ConnId
   bytes :- ByteArray
   proc-type :- ProcType]
  (let [state (state/get-conn-state db conn-id)
        f (case state
            :start (throw-far-error
                    "Illegal state. Data rcvd before connection."
                    :execution-error :data-rcvd-before-connection
                    (sym-map state conn-id proc-type))
            :connected bytes->commands-connected
            bytes->commands*)]
    (if state
      (f db conn-id bytes proc-type)
      [])))
