(ns farbetter.mu.mock-net
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [go-safe inspect sym-map]
                         :cljs [byte-array long])]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [go-safe inspect sym-map]])))


(defprotocol INetwork
  (add-server [this server-url on-connect])
  (remove-server [this server-url])
  (connect-client
    [this client-id server-url on-rcv]
    [this client-id server-url on-rcv opts]
    "Connect a client to a server via the mock network."))

(defrecord MockNet [url->server-on-connect client-id->disconnectors]
  INetwork
  (add-server [this server-url on-connect]
    (swap! url->server-on-connect assoc server-url on-connect))
  (remove-server [this server-url]
    (swap! url->server-on-connect dissoc server-url))
  (connect-client [this client-id server-url on-rcv]
    (connect-client this client-id server-url on-rcv {}))
  (connect-client [this client-id server-url on-rcv opts]
    (let [svr-on-connect (@url->server-on-connect server-url)
          _ (when-not svr-on-connect
              (u/throw-far-error "Bad server URL" :illegal-argument
                                 :bad-server-url (sym-map server-url)))
          {:keys [on-connect on-disconnect on-error connect-ms]
           :or {connect-ms 1}} opts
          closer (fn []
                   (doseq [disconnector (@client-id->disconnectors client-id)]
                     (when disconnector
                       (disconnector)))
                   (swap! client-id->disconnectors dissoc client-id))
          ;; TODO: Store on-error handlers and
          ;; call them when appropriate
          ret (svr-on-connect client-id on-rcv closer)
          sender #(try
                    ((:on-rcv ret) %)
                    (catch #?(:clj Exception :cljs :default) e
                      (when on-error
                        (on-error e))))]
      (swap! client-id->disconnectors assoc client-id
             [(:on-disconnect ret) on-disconnect])
      (when on-connect
        (go-safe
         (ca/<! (ca/timeout connect-ms))
         (on-connect)))
      (sym-map closer sender))))

(defn make-mock-net []
  (let [url->server-on-connect (atom {})
        client-id->disconnectors (atom {})]
    (map->MockNet (sym-map url->server-on-connect client-id->disconnectors))))
