(ns farbetter.mu.client
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca
    :refer [#? (:clj go)]]
   [farbetter.mu.msgs :as msgs :refer [APISpec]]
   [farbetter.mu.proc :as proc]
   [farbetter.mu.state :as state]
   [farbetter.mu.transport :as tp]
   [farbetter.mu.utils :as mu :refer [Channel ConnId RequestId UserId]]
   [farbetter.mu.websocket :as websocket]
   [farbetter.pete :as pete]
   [farbetter.roe :as roe]
   [farbetter.roe.schemas :refer [AvroData]]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [go-safe inspect sym-map])]]
   [freedomdb.frontend :as fdb]
   [freedomdb.schemas :refer [DBType]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [cljs.core.async.macros :refer [go]]
      [farbetter.utils :as u :refer [go-safe inspect sym-map]])))

(declare add-rpc apis->api-map check-user-id gc-rpcs handle-login-rs
         handle-rpc-rs handle-rq-failed map->Client modify-db
         start-connection-loop)

(def rcv-chan-buf-size 100)
(def wait-ms (* 1000 1))
(def bad-login-kw :bad-login)
(def max-rpc-wait-ms (* 1000 20))

(defprotocol IClient
  (logged-in? [this])
  (call-rpc
    [this svc-name fn-name args on-success on-failure]
    [this svc-name fn-name args on-success on-failure timeout-ms])
  (stop [this] "Stop processing loop. Useful for testing."))

(s/defn make-client :- (s/protocol IClient)
  "Create a mu client.
   Arguments:
   - apis"
  [apis :- [APISpec]
   email :- s/Str
   password :- s/Str
   gw-urls-factory :- (s/=> [s/Str])
   conn-factory :- (s/=> s/Any)]
  (let [db-atom (atom (state/make-db apis modify-db))
        api-map (apis->api-map apis)
        user-id-atom (atom nil)
        active?-atom (atom true)
        rcv-chan (ca/chan rcv-chan-buf-size)
        repeater (pete/make-repeater)
        client (map->Client (sym-map db-atom api-map user-id-atom active?-atom
                                     repeater))
        send-login (fn [db conn-id]
                     (let [msg (sym-map email password)]
                       (state/send-msg db conn-id :client-login-rq msg)))
        process-msg (fn [db conn-id msg-name msg]
                      (case msg-name
                        :client-login-rs (handle-login-rs db conn-id msg
                                                          user-id-atom)
                        :rpc-rs (handle-rpc-rs db conn-id msg)
                        :rq-failed (handle-rq-failed db conn-id msg)
                        (throw-far-error
                         "Got unknown msg type."
                         :execution-error :rcvd-unknown-msg-type
                         (sym-map msg-name msg conn-id))))
        addl-op->f (sym-map process-msg send-login)]
    (proc/start-procs repeater active?-atom db-atom rcv-chan :cl addl-op->f
                      gc-rpcs)
    (start-connection-loop active?-atom db-atom rcv-chan gw-urls-factory
                           conn-factory)
    client))

(defrecord Client [db-atom api-map user-id-atom active?-atom repeater]
  IClient
  (logged-in? [this]
    (check-user-id user-id-atom)
    (boolean (state/get-logged-in-conn-ids @db-atom)))
  (call-rpc [this svc-name fn-name arg on-success on-failure]
    (call-rpc this svc-name fn-name arg on-success on-failure
              mu/default-rpc-timeout-ms))
  (call-rpc [this svc-name fn-name arg on-success on-failure timeout-ms]
    (check-user-id user-id-atom)
    (let [timeout-ms (min timeout-ms max-rpc-wait-ms)
          api (api-map svc-name)
          service-api-version (select-keys api [:service-name :major-version])
          user-id @user-id-atom
          request-id (-> (u/make-v1-uuid)
                         (u/uuid->int-map))
          schemas (get-in api [:fn-schemas fn-name])
          {:keys [arg-schema return-schema]} schemas
          arg-schema-fp (state/schema->fingerprint @db-atom arg-schema)
          encoded-arg (roe/edn->avro-byte-array arg-schema arg)
          msg (sym-map service-api-version user-id request-id fn-name
                       arg-schema-fp encoded-arg timeout-ms)
          conn-id (state/get-gw-conn-id @db-atom)]
      (if conn-id
        (swap! db-atom
               #(-> %
                    (state/send-msg conn-id :rpc-rq msg)
                    (add-rpc request-id return-schema on-success on-failure)))
        (on-failure :not-logged-in))
      nil))
  (stop [this]
    (reset! active?-atom false)
    (pete/stop repeater)))

;;;;;;;;;;;;;;;;;;;; Helper fns ;;;;;;;;;;;;;;;;;;;;

(defn- check-user-id [user-id-atom]
  (when (= bad-login-kw @user-id-atom)
    (throw-far-error "Login failed due to bad credentials."
                     :execution-error :login-failed-bad-credentials {})))

(defn apis->api-map [apis]
  (reduce (fn [acc api]
            (let [{:keys [service-name]} api]
              (assoc acc service-name api)))
          {} apis))

(s/defn start-connection-loop :- nil
  [active?-atom :- (s/atom s/Bool)
   db-atom :- (s/atom DBType)
   rcv-chan :- Channel
   gw-urls-factory :- (s/=> Channel)
   conn-factory :- (s/=> s/Any)]
  (go
    (try
      (while @active?-atom
        (let [db @db-atom
              li-conn-ids (state/get-logged-in-conn-ids db)
              make-conns! (fn make-conns! []
                            (go
                              (let [[status gw-urls] (ca/<! (gw-urls-factory))]
                                (case status
                                  :success (doseq [url gw-urls]
                                             (tp/connect-to-gw
                                              db-atom url rcv-chan
                                              conn-factory))
                                  :failure nil
                                  (throw-far-error
                                   (str "Bad status `" status "` returned.")
                                   :execution-error :bad-status
                                   (sym-map status))))))]
          (case (count li-conn-ids)
            0 (let [nli-conn-ids (state/get-non-logged-in-conn-ids db)]
                (if (pos? (count nli-conn-ids))
                  (when (every? #(state/timed-out? db %) nli-conn-ids)
                    (swap! db-atom state/close-conns nli-conn-ids)
                    (make-conns!))
                  (make-conns!)))
            1 (debugf "Branch 1")
            (swap! db-atom state/prune-conns))
          (ca/<! (ca/timeout wait-ms))))
      (catch #?(:clj Exception :cljs :default) e
        (debugf "Got exception: %s" e)
        (u/log-exception e))))
  nil)

(defn handle-rpc-rs [db conn-id msg]
  (debugf "Client got rpc-rs. msg: %s" msg)
  (let [{:keys [request-id return-schema-fp]} msg
        w-schema (state/fingerprint->schema db return-schema-fp)]
    (if w-schema
      (let [row (fdb/select-one db {:tables [:client-rpcs]
                                    :where [:= :request-id request-id]})
            {:keys [on-success return-schema]} row
            {:keys [encoded-return-value]} msg
            ret (roe/avro-byte-array->edn w-schema return-schema
                                          encoded-return-value)]
        (on-success ret)
        (fdb/delete db :client-rpcs [:= :request-id request-id]))
      (let [fingerprint (roe/edn-schema->fingerprint msgs/rpc-rs-schema)
            fragment-data (roe/edn->avro-byte-array msgs/rpc-rs-schema msg)
            fragment (sym-map fingerprint fragment-data)
            bytes (roe/edn->avro-byte-array msgs/fragment-schema fragment)]
        (state/handle-missing-schema db conn-id return-schema-fp bytes)))))

(defn handle-rq-failed [db conn-id msg]
  (debugf "Client got rq-failed. msg: %s" msg)
  (let [{:keys [request-id reason]} msg
        on-failure (fdb/select-one db {:tables [:client-rpcs]
                                       :fields :on-failure
                                       :where [:= :request-id request-id]})]
    (on-failure [request-id reason])
    (fdb/delete db :client-rpcs [:= :request-id request-id])))

(s/defn handle-login-rs :- DBType
  [db :- DBType
   conn-id :- ConnId
   msg :- AvroData
   user-id-atom :- (s/atom (s/maybe UserId))]
  (let [{:keys [was-successful user-id]} msg]
    (if was-successful
      (do
        (reset! user-id-atom user-id)
        (state/set-conn-state db conn-id :logged-in))
      (throw-far-error "Client login failed due to invalid credentials."
                       :execution-error :login-failed-invalid-creds
                       (sym-map conn-id)))))

(s/defn add-rpc :- DBType
  [db :- DBType
   request-id :- RequestId
   return-schema :- AvroData
   on-success :- (s/=> s/Any)
   on-failure :- (s/=> s/Any)]
  (let [insert-time-ms (u/long (u/get-current-time-ms))]
    (fdb/insert db :client-rpcs (sym-map request-id return-schema on-success
                                         on-failure insert-time-ms))))

(s/defn modify-db :- DBType
  [db :- DBType]
  (fdb/create-table db
                    :client-rpcs {:request-id {:type :any :indexed true}
                                  :return-schema {:type :any :indexed false}
                                  :on-success {:type :any :indexed false}
                                  :on-failure {:type :any :indexed false}
                                  :insert-time-ms {:type :int8 :indexed true}}))

(s/defn gc-rpcs :- DBType
  [db :- DBType]
  (let [max-time-ms (#?(:clj - :cljs .sub) (u/long (u/get-current-time-ms))
                       max-rpc-wait-ms)]
    (fdb/delete db :client-rpcs [:< :insert-time-ms max-time-ms])))
