(ns farbetter.freedomdb.transit
  (:require
   [#?(:clj clj-time.format :cljs cljs-time.format) :as f]
   [clojure.data.avl :as avl]
   [cognitect.transit :as transit]
   [farbetter.freedomdb.mem-row-store :as mrs]
   [farbetter.freedomdb.schemas :as sch]
   [farbetter.utils :as u :refer
    [#?@(:clj [inspect sym-map]) throw-far-error]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf infof warnf errorf]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])
     :clj
     (:import [java.io ByteArrayInputStream ByteArrayOutputStream]
              [org.joda.time DateTime])))

(declare transit->edn)

(def initial-transit-buffer-size 4096)
(def date-time-formatter (f/formatters :date-hour-minute-second-ms))
(def date-time-transit-tag "dt")
(def mem-row-store-transit-tag "mrs")
(def avl-sorted-map-transit-tag "asm")

(def date-time-writer
  (transit/write-handler
   (constantly date-time-transit-tag)
   #(f/unparse date-time-formatter %)))

(def date-time-reader
  (transit/read-handler
   #(f/parse date-time-formatter %)))

(s/defn edn->transit :- (s/maybe s/Str)
  [edn :- s/Any
   addl-handlers :- sch/TransitWriteHandlersMap]
  (let [avl-writer (transit/write-handler
                    (constantly avl-sorted-map-transit-tag)
                    #(-> (seq %)
                         (flatten)
                         (edn->transit addl-handlers)))
        mrs-writer (transit/write-handler
                    (constantly mem-row-store-transit-tag)
                    #(as-> % x
                       (seq x)
                       (flatten x)
                       (apply hash-map x)
                       (edn->transit x addl-handlers)))
        base-handlers {clojure.data.avl.AVLMap avl-writer
                       farbetter.freedomdb.mem_row_store.MemRowStore
                       mrs-writer}
        handlers (merge base-handlers addl-handlers)]
    #?(:clj
       (let [out (ByteArrayOutputStream. initial-transit-buffer-size)
             writer (transit/writer
                     out :json
                     {:handlers (merge handlers
                                       {DateTime date-time-writer})})]
         (transit/write writer edn)
         (.toString ^ByteArrayOutputStream out "UTF-8"))
       :cljs
       (transit/write (transit/writer
                       :json
                       {:handlers (merge handlers
                                         {UtcDateTime date-time-writer})})
                      edn))))

(s/defn transit->edn :- s/Any
  [transit-str :- (s/maybe s/Str)
   addl-handlers :- sch/TransitReadHandlersMap]
  (when transit-str
    (let [mem-row-store-reader (transit/read-handler
                                #(let [m (transit->edn % addl-handlers)]
                                   (mrs/map->MemRowStore m)))
          avl-sorted-map-reader (transit/read-handler
                                 #(apply avl/sorted-map
                                         (transit->edn % addl-handlers)))
          base-handlers {date-time-transit-tag date-time-reader
                         mem-row-store-transit-tag mem-row-store-reader
                         avl-sorted-map-transit-tag avl-sorted-map-reader}
          handlers (merge base-handlers addl-handlers)]
      #?(:clj
         (let [bytes (.getBytes transit-str "UTF-8")
               in (ByteArrayInputStream. bytes)
               reader (transit/reader in :json (sym-map handlers))]
           (transit/read reader))
         :cljs
         (transit/read (transit/reader :json (sym-map handlers))
                       transit-str)))))
