(ns farbetter.freedomdb.row-store
  (:require
   [farbetter.freedomdb.schemas :refer
    [DBTableMetadata DB FieldAttrsMap FieldName FieldsMap
     FieldType GetAllRowsOutputStyle GetAllRowsReturn
     InequalityOperator RowId TableName TypeMap ValueMap
     Value]]
   [farbetter.utils :refer [#?@(:clj [inspect sym-map]) throw-far-error]]
   [schema.core :as s :include-macros true]))

(defprotocol RowStore
  (create-table- [this table-name fields-map])
  (drop-table- [this table-name])
  (get-metadata- [this table-name])
  (get-table-names-set- [this])
  (add-field- [this table-name field-name attrs])
  (drop-field- [this table-name field-name])
  (modify-field- [this table-name field-name attrs])
  (get-row-count- [this table-name])
  (get-row- [this table-name row-id])
  (get-all-rows- [this table-name output-fmt])
  (get-row-ids-eq- [this table-name field-name v v-type])
  (get-row-ids-ineq- [this table-name field-name v v-type op])
  (put-row- [this table-name val-map type-map indexed-fields])
  (update-row- [this table-name row-id old-val-map new-val-map
                type-map indexed-fields])
  (delete-row- [this table-name row-id type-map])
  (delete-all-rows- [this table-name type-map]))

(s/defn get-metadata :- (s/maybe DBTableMetadata)
  [db :- DB
   table-name :- TableName]
  (get-metadata- db table-name))

(s/defn create-table :- DB
  [db :- DB
   table-name :- TableName
   fields-map :- FieldsMap]
  (create-table- db table-name fields-map))

(s/defn drop-table
  [db :- DB
   table-name :- TableName]
  (drop-table- db table-name))

(s/defn get-table-names-set :- #{TableName}
  [db :- DB]
  (get-table-names-set- db))

(s/defn add-field :- DB
  [db :- DB
   table-name :- TableName
   field-name :- FieldName
   attrs :- FieldAttrsMap]
  (add-field- db table-name field-name attrs))

(s/defn drop-field :- DB
  [db :- DB
   table-name :- TableName
   field-name :- FieldName]
  (drop-field- db table-name field-name))

(s/defn modify-field :- DB
  [db :- DB
   table-name :- TableName
   field-name :- FieldName
   attrs :- FieldAttrsMap]
  (modify-field- db table-name field-name attrs))

(s/defn get-row-count :- s/Num
  [db :- DB
   table-name :- TableName]
  (get-row-count- db table-name))

(s/defn get-row :- ValueMap
  [db :- DB
   table-name :- TableName
   row-id :- RowId]
  (get-row- db table-name row-id))

(s/defn get-all-rows :- GetAllRowsReturn
  [db :- DB
   table-name :- TableName
   output-fmt :- GetAllRowsOutputStyle]
  (get-all-rows- db table-name output-fmt))

(s/defn get-row-ids-eq :- (s/maybe #{RowId})
  [db :- DB
   table-name :- TableName
   field-name :- FieldName
   v :- Value
   v-type :- FieldType]
  (get-row-ids-eq- db table-name field-name v v-type))

(s/defn get-row-ids-ineq :- (s/maybe #{RowId})
  [db :- DB
   table-name :- TableName
   field-name :- FieldName
   v :- Value
   v-type :- FieldType
   op :- InequalityOperator]
  (get-row-ids-ineq- db table-name field-name v v-type op))

(s/defn put-row :- DB
  [db :- DB
   table-name :- TableName
   val-map :- ValueMap
   type-map :- TypeMap
   indexed-fields :- #{FieldName}]
  (put-row- db table-name val-map type-map indexed-fields))

(s/defn update-row :- DB
  [db :- DB
   table-name :- TableName
   row-id :- RowId
   old-val-map :- ValueMap
   new-val-map :- ValueMap
   type-map :- TypeMap
   indexed-fields :- #{FieldName}]
  (update-row- db table-name row-id old-val-map new-val-map
               type-map indexed-fields))

(s/defn delete-row :- DB
  [db :- DB
   table-name :- TableName
   row-id :- RowId
   type-map :- TypeMap]
  (delete-row- db table-name row-id type-map))

(s/defn delete-all-rows :- DB
  [db :- DB
   table-name :- TableName
   type-map :- TypeMap]
  (delete-all-rows- db table-name type-map))
