(ns freedomdb.kv-store
  (:refer-clojure :exclude [filter get keys remove])
  (:require
   [farbetter.utils :as u :refer [#?@(:clj [inspect sym-map]) throw-far-error]]
   [schema.core :as s :include-macros true])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])))

(declare check-filter-options check-keyspace)

(def valid-keyspaces #{:metadata :rvi :vri})
(def valid-output-styles #{:keys-and-values :keys-only :values-only})
(def valid-filter-ops #{:< :<= :> :>=})

(defprotocol KVStore
  (get- [this keyspace k])
  (put- [this keyspace k v])
  (remove- [this keyspace k])
  (keys- [this keyspace])
  (filter- [this keyspace options]))

(def KVKeyType s/Str)
(def KVValueType s/Str)
(def KVKeyOrValueType s/Str)
(def KVMapEntryType [(s/one KVKeyType "key") (s/one KVValueType "value")])
(def KVStoreType (s/protocol KVStore))
(def KVKeyspaceType (apply s/enum valid-keyspaces))
(def KVFilterOpType (apply s/enum valid-filter-ops))
(def KVFilterOutputStyleType (apply s/enum valid-output-styles))
(def KVFilterOptionsType
  {(s/required-key :output) KVFilterOutputStyleType
   (s/required-key :prefix) KVKeyType
   (s/optional-key :filter-op) KVFilterOpType
   (s/optional-key :filter-val) KVValueType})
(def KVFilterReturnType
  (s/maybe [(s/if sequential? KVMapEntryType KVKeyOrValueType)]))

;;;;;;;;; API ;;;;;;;;;

(s/defn get :- (s/maybe KVValueType)
  "Get the value of key `k` in the given keyspace."
  [store :- KVStoreType
   keyspace :- KVKeyspaceType
   k :- KVKeyType]
  (get- store keyspace k))

(s/defn put :- KVStoreType
  "Store the value `v` at key `k` in the given keyspace."
  [store :- KVStoreType
   keyspace :- KVKeyspaceType
   k :- KVKeyType
   v :- KVValueType]
  (put- store keyspace k v))

(s/defn remove :- KVStoreType
  "Remove the key `k` from the given keyspace."
  [store :- KVStoreType
   keyspace :- KVKeyspaceType
   k :- KVKeyType]
  (remove- store keyspace k))

(s/defn keys :- (s/maybe [KVKeyType])
  "Returns a (possibly lazy) seq of the keys of the given keyspace."
  [store :- KVStoreType
   keyspace :- KVKeyspaceType]
  (keys- store keyspace))

(s/defn filter :- KVFilterReturnType
  "Return a subset of keys and/or values from the given keyspace."
  [store :- KVStoreType
   keyspace :- KVKeyspaceType
   options :- KVFilterOptionsType]
  (filter- store keyspace options))
