(ns freedomdb.schemas
  (:require
   [clojure.set :refer [union]]
   [schema.core :as s :include-macros true]))

(declare ExpressionType FieldNameType JoinExpressionOrClauseType
         OrderByDirectionType)

;;;;;;;;; Helper fns ;;;;;;;;;

(defn op-matches? [op-type]
  #(nil? (s/check op-type (first %))))

(defn valid-order-by? [expr]
  (and (sequential? expr)
       (even? (count expr))
       (loop [[field-name direction & rest] expr
              result false]
         (if (nil? direction)
           result
           (if (and (nil? (s/check FieldNameType field-name))
                    (nil? (s/check OrderByDirectionType direction)))
             (recur rest true)
             false)))))

;;;;;;;;; Frontend Schemas ;;;;;;;;;

(def ValueType (s/conditional
                integer? s/Int
                string? s/Str
                keyword? s/Keyword
                true? s/Bool
                false? s/Bool
                (constantly true) s/Any))
(def FieldTypeType (s/enum :int4 :str1000 :kw :bool :any))
(def TableNameType s/Keyword)
(def FieldNameType s/Keyword)
(def NotOperatorType (s/eq :not))
(def EqualityOperatorType (s/eq :=))
(def InequalityOperatorType (s/enum :< :> :<= :>=))
(def CombinationOperatorType (s/enum :and :or))
(def JoinTypeType (s/enum :inner :left :right :full
                          :left-outer :right-outer :full-outer))
(def JoinExpressionType
  [(s/one EqualityOperatorType :=)
   (s/one TableNameType :left-table)
   (s/one TableNameType :right-table)])
(def CombinationJoinType
  [(s/one CombinationOperatorType :combination-operator)
   (s/recursive #'JoinExpressionOrClauseType)])
(def NotJoinType
  (s/pair NotOperatorType :not-operator
          (s/recursive #'JoinExpressionOrClauseType) :clause))
(def JoinExpressionOrClauseType
  (s/conditional
   (op-matches? EqualityOperatorType) JoinExpressionType
   (op-matches? CombinationOperatorType) CombinationJoinType
   (op-matches? NotOperatorType) NotJoinType))
(def JoinMapType
  {(s/required-key :on) JoinExpressionType
   (s/required-key :type) JoinTypeType})
(def JoinClauseType (s/conditional
                     map? JoinMapType
                     sequential? JoinExpressionOrClauseType
                     ;;sequential? JoinExpressionType
                     ))
(def ValueMapType {FieldNameType ValueType})
(def AsKeywordType (s/eq :as))
(def AggregateFnKeywordType (s/enum :sum :count :min :max))
(def AggregateType [(s/one AggregateFnKeywordType :agg-fn)
                    (s/one FieldNameType :field-name)
                    (s/one AsKeywordType :as-keyword)
                    (s/one FieldNameType :name)])
(def FieldAttrsMapType
  {(s/required-key :type) FieldTypeType
   (s/optional-key :indexed) s/Bool
   (s/optional-key :default) ValueType})
(def ModifyFieldAttrsMapType
  {(s/optional-key :type) FieldTypeType
   (s/optional-key :indexed) s/Bool
   (s/optional-key :default) ValueType})
(def FieldsMapType {FieldNameType FieldAttrsMapType})
(def RowIdType s/Num)
(def RowIdsOrAllType (s/if keyword? (s/eq :all) #{RowIdType}))
(def FieldNamesOrAllType (s/if keyword? (s/eq :all) #{FieldNameType}))
(def OrderByDirectionType (s/enum :asc :desc))
(def OrderByClauseType (s/pred valid-order-by?))
(def BinaryOperatorType (s/if #(= := %)
                          EqualityOperatorType
                          InequalityOperatorType))
(def ScanFnOperatorType (s/enum :scan-fn))
(def ScanFnType (s/=> s/Bool & [ValueType]))
(def NotExpressionType (s/pair NotOperatorType :not-operator
                               (s/recursive #'ExpressionType) :expression))
(def BinaryExpressionType
  [(s/one BinaryOperatorType :binary-operator)
   (s/one FieldNameType :field-name)
   (s/one ValueType :argument)])
(def CombinationExpressionType
  [(s/one CombinationOperatorType
          :combination-operator)
   (s/recursive #'ExpressionType)])
(def ScanFnExpressionType [(s/one ScanFnOperatorType :scan-fn-operator)
                           (s/one ScanFnType :scan-function)
                           FieldNameType])
(def ExpressionType
  (s/conditional
   (op-matches? NotOperatorType) NotExpressionType
   (op-matches? BinaryOperatorType) BinaryExpressionType
   (op-matches? CombinationOperatorType) CombinationExpressionType
   (op-matches? ScanFnOperatorType) ScanFnExpressionType))
(def SelectQueryType
  {(s/required-key :tables) (s/if sequential?
                              [TableNameType]
                              TableNameType)
   (s/optional-key :fields) (s/maybe
                             (s/if sequential?
                               [FieldNameType]
                               FieldNameType))
   (s/optional-key :join) JoinClauseType
   (s/optional-key :where) ExpressionType
   (s/optional-key :aggregate) AggregateType
   (s/optional-key :order-by) OrderByClauseType})
(def SelectOneReturnType (s/maybe
                          (s/conditional
                           map? ValueMapType
                           sequential? [ValueType]
                           :else ValueType)))
(def SelectReturnType (s/maybe [SelectOneReturnType]))
(def UpdateQueryType
  {(s/required-key :set) ValueMapType
   (s/optional-key :where) ExpressionType})
(def TypeMapType {FieldNameType FieldTypeType})
(def DBTableMetadataType
  {(s/required-key :table-name) TableNameType
   (s/required-key :all-fields) #{FieldNameType}
   (s/required-key :indexed-fields) #{FieldNameType}
   (s/required-key :type-map) TypeMapType
   (s/required-key :defaults) {FieldNameType ValueType}})
(def DBType (s/pred associative?))

;;;;;;;;; Backend Schemas ;;;;;;;;;

(def RowIdValueMapPair [(s/one RowIdType :row-id)
                        (s/one ValueMapType :value-map)])
(def GetAllRowsOutputStyleType (s/enum :row-ids-only
                                       :value-maps-only
                                       :row-ids-and-value-maps))
(def GetAllRowsReturnType [(s/conditional
                            sequential? RowIdValueMapPair
                            number? RowIdType
                            map? ValueMapType)])
