(ns farbetter.chumbawamba
  (:refer-clojure :exclude [send])
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async)
    :refer [<! timeout]]
   [farbetter.chumbawamba.keep-alive :as ka]
   [farbetter.chumbawamba.websocket :as ws]
   [farbetter.utils :refer
    [get-current-time-ms throw-far-error #?@(:clj [go-safe inspect sym-map])]]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [go-safe inspect sym-map]])))

(def default-keepalive-secs 30)
(def min-reconnect-ms 1000)
(def default-max-reconnect-secs 30)
(def backoff-factor 1.3)

(defprotocol Sendable
  (send [this string-or-bytes] "Send a string or bytes"))

(defprotocol ConnectionManager
  (connect [this] "Connect the websocket")
  (close [this] "Close the websocket")
  (get-connection-state [this]
                        "Returns either :connected, connecting, or :closed")
  (-on-connect-impl [this] "Internal use only.")
  (-on-close-impl [this reason] "Internal use only.")
  (-set-keep-alive [this keep-alive-rec]))

(defrecord ResilientWS [uri websocket connection-state on-rcv on-connect
                        on-close reconnect-policy max-reconnect-secs
                        configured-reconnect-policy keep-alive]
  Sendable
  (send [this string-or-bytes]
    (let [end-ms (+ (get-current-time-ms)
                    (* 1000 max-reconnect-secs))]
      (go-safe
       (loop []
         (cond
           (= :connected @connection-state)
           (do
             (ws/ws-send @websocket string-or-bytes)
             (ka/reset-keep-alive @keep-alive)
             [:success :data-sent])

           (< (get-current-time-ms) end-ms)
           (do
             (connect this)
             (<! (timeout 10))
             (recur))

           :else
           [:failure :could-not-connect])))))

  ConnectionManager
  (connect [this]
    (when (= :closed @connection-state)
      (reset! connection-state :connecting)
      (reset! reconnect-policy configured-reconnect-policy)
      (let [end-time-ms (+ (get-current-time-ms)
                           (* 1000 max-reconnect-secs))]
        (go-safe
         (loop [wait-ms 0]
           (let [ws-chan (ws/make-websocket
                          uri (partial -on-close-impl this) on-rcv)
                 [status ret] (<! ws-chan)
                 new-ms #(cond
                           (= :aggressive @reconnect-policy) 0
                           (zero? wait-ms) min-reconnect-ms
                           :else (* backoff-factor wait-ms))]
             (cond
               (= :success status)
               (do
                 (reset! websocket ret)
                 (-on-connect-impl this))

               (or (= :none @reconnect-policy)
                   (>= (get-current-time-ms) end-time-ms))
               (reset! connection-state :closed)

               :else
               (let [ms (-> (rand-int wait-ms)
                            (+ min-reconnect-ms))]
                 (<! (timeout ms))
                 (recur (new-ms))))))))))

  (close [this]
    (reset! reconnect-policy :none)
    (when (= :connected @connection-state)
      (ka/stop @keep-alive)
      (ws/ws-close @websocket)))

  (get-connection-state [this]
    @connection-state)

  (-on-connect-impl [this]
    (reset! connection-state :connected)
    (infof "Websocket to %s connected." uri)
    (when on-connect
      (on-connect this))
    (ka/start @keep-alive))

  (-on-close-impl [this reason]
    (reset! websocket nil)
    (reset! connection-state :closed)
    (ka/stop @keep-alive)
    (infof "Websocket to %s closed. Reason: %s" uri reason)
    (when on-close
      (on-close this reason))
    (when (not= :none @reconnect-policy)
      (connect this)))

  (-set-keep-alive [this keep-alive-rec]
    (reset! keep-alive keep-alive-rec)))

(defn check-reconnect-policy [reconnect-policy]
  (when-not (contains? #{:backoff :aggressive :none} reconnect-policy)
    (throw-far-error (str "Illegal reconnect-policy: " reconnect-policy)
                     :illegal-argument :illegal-reconnect-policy
                     (sym-map reconnect-policy))))

(defn make-resilient-ws
  "Make a resilient websocket connection that reconnects on socket closure.
   Arguments:
   - uri - URI of the websocket server. Required.
   - keep-alive-msg - A message to be sent periodically to keep the connection
        alive. Required. May be a string or a byte array.
   - on-rcv - A callback function of one argument (data) to be
        called on receipt of text or binary data. Required.
   - opts - key-value pairs. Optional. Possible keys:
        - :on-connect - Callback function of one argument (this) to be called
             on successful connection (or reconnection) of the websocket
        - :on-close - Callback function of one argument (this) to be called
             whenever the websocket connection closes.
        - :keep-alive-secs - Number of seconds between sending keep-alive
             messages. Optional. Defaults to 30.
        - :reconnect-policy - A keyword argument specifying how to handle
             connection closures. Optional. One of:
             - :backoff - Try to reconnect at exponentially longer random
                  intervals. Gives up after max-reconnect-secs. This is the
                  default if no policy is specified.
             - :aggressive - Try to reconnect aggressively. Never stop trying.
             - :none - Don't try to reconnect.
        - :max-reconnect-secs - Number of seconds to try to reconnect. Optional.
             Defaults to 60."
  [uri keep-alive-msg on-rcv &
   {:keys [on-connect
           on-close
           keep-alive-secs
           reconnect-policy
           max-reconnect-secs]
    :or {keep-alive-secs default-keepalive-secs
         reconnect-policy :backoff
         max-reconnect-secs default-keepalive-secs}}]
  (let [websocket (atom nil)
        connection-state (atom :closed)
        reconnect-policy (atom reconnect-policy)
        configured-reconnect-policy reconnect-policy
        keep-alive (atom nil)
        args (sym-map uri on-rcv on-connect on-close keep-alive
                      reconnect-policy connection-state
                      configured-reconnect-policy max-reconnect-secs websocket)
        _ (check-reconnect-policy @(:reconnect-policy args))
        rws (map->ResilientWS args)
        keep-alive-fn #(send rws keep-alive-msg)
        keep-alive (ka/make-keep-alive (* 1000 keep-alive-secs) keep-alive-fn)]
    (-set-keep-alive rws keep-alive)
    (connect rws)
    rws))
