package jre2;


public class Matcher {
	private JRE2 jre2;
	private String input = null;
	private StringUTF8 inputUTF8;

	private int[] groups = null;
	private boolean findsFinished = false;

	private int prevEnd = 0;


	// constructors are not public but have package access so they're accessible by JRE2

	Matcher(JRE2 jre2, String input) {
		this.jre2 = jre2;
		this.input = input;
		inputUTF8 = new StringUTF8(input);
	}

	Matcher(JRE2 jre2, StringUTF8 inputUTF8) {
		this.jre2 = jre2;
		this.inputUTF8 = inputUTF8;
	}


	/**
	 * Frees the native RE2 object.
	 */
	public void close() {
		jre2.close();
	}


	/**
	 * Attempts to match the entire input string.
	 * If successful, group(int index) will return the value of a capturing group.
	 * @return true iff a successful match.
	 * @throws JRE2Exception
	 */
	public boolean matches() throws JRE2Exception {
		groups = jre2.fullMatch(inputUTF8);
		if (groups == null) {
			return false;
		}
		return true;
	}


	/**
	 * Attempts to find a match starting at the end of the previous successful match, or zero if no previous match.
	 * If successful, group() will return the match, and group(int index) will return the value of a capturing group.
	 * @return true iff a successful match.
	 * @throws JRE2Exception
	 */
	public boolean find() throws JRE2Exception {
		if (findsFinished) return false;

		int offset = 0;
		if (groups != null) {
			offset = groupOffset(0) + groupLength(0);
		}

		int[] newGroups = jre2.partialMatch(inputUTF8, offset);
		if (newGroups == null) {
			findsFinished = true;
			return false;
		}
		groups = newGroups;
		return true;
	}


	public String group() {
		return groupUTF8().toString();
	}

	public String group(int index) {
		StringUTF8 sUTF8 = groupUTF8(index);
		if (sUTF8 == null) {
			return null;
		} else {
			return sUTF8.toString();
		}
	}

	public int groupCount() {
		return jre2.groupCount();
	}


	public StringUTF8 groupUTF8() {
		if (groups == null) {
			throw new IllegalStateException("No previous successful match");
		}
		return new StringUTF8(inputUTF8, groupOffset(0), groupLength(0));
	}

	public StringUTF8 groupUTF8(int index) {
		if (groups == null) {
			throw new IllegalStateException("No previous successful match");
		}
		if (index < 0 || index * 2 >= groups.length) {
			throw new IndexOutOfBoundsException("No match group with index " + index);
		}
		int offset = groupOffset(index);

		// Return null for an optional group, e.g. (foo)?, that didn't match.
		if (offset < 0) {
			return null;
		} else {
			return new StringUTF8(inputUTF8, offset, groupLength(index));
		}
	}


	public int start() {
		if (groups == null) {
			throw new IllegalStateException("No previous successful match");
		}
		return groupOffset(0);
	}

	public int end() {
		if (groups == null) {
			throw new IllegalStateException("No previous successful match");
		}
		return groupOffset(0) + groupLength(0);
	}


	private int groupOffset(int index) {
		return groups[index * 2] - inputUTF8.offset();
	}

	private int groupLength(int index) {
		return groups[index * 2 + 1];
	}


	public Matcher reset() {
		groups = null;
		findsFinished = false;
		prevEnd = 0;
		return this;
	}

	public String replaceAll(String replacement) throws JRE2Exception {
		StringUTF8 ret = replaceAll(new StringUTF8(replacement));
		if (ret == inputUTF8 && input != null) {
			return input;
		} else {
			return ret.toString();
		}
	}

	public String replaceFirst(String replacement) throws JRE2Exception {
		StringUTF8 ret = replaceFirst(new StringUTF8(replacement));
		if (ret == inputUTF8 && input != null) {
			return input;
		} else {
			return ret.toString();
		}
	}

	// derived from the code for java.util.regex.Matcher.replaceAll
	public StringUTF8 replaceAll(StringUTF8 replacement) throws JRE2Exception {
		reset();
		boolean result = find();
		if (result) {
			StringUTF8Buffer sb = new StringUTF8Buffer();
			do {
				appendReplacement(sb, replacement);
				result = find();
			} while (result);
			appendTail(sb);
			return sb.toStringUTF8();
		}
		return inputUTF8;
	}

	// derived from the code for java.util.regex.Matcher.replaceFirst
	public StringUTF8 replaceFirst(StringUTF8 replacement) throws JRE2Exception {
		reset();
		if (!find())
			return inputUTF8;
		StringUTF8Buffer sb = new StringUTF8Buffer();
		appendReplacement(sb, replacement);
		appendTail(sb);
		return sb.toStringUTF8();
	}

	// derived from the code for java.util.regex.Matcher.appendReplacement
	public void appendReplacement(StringUTF8Buffer sb, StringUTF8 replacement) {
		// Append the intervening text
		sb.append(new StringUTF8(inputUTF8, prevEnd, start() - prevEnd));

		// Process substitution string to replace group references with groups
		byte[] bytes = replacement.bytes();
		int cursor = replacement.offset();

		while (cursor < replacement.length()) {
			byte nextByte = bytes[cursor];
			if (nextByte == '\\') {
				cursor++;
				nextByte = bytes[cursor];
				sb.append(nextByte);
				cursor++;
			} else if (nextByte == '$') {
				// Skip past $
				cursor++;
				// The first number is always a group
				int refNum = (int)bytes[cursor] - '0';
				if ((refNum < 0)||(refNum > 9))
					throw new IllegalArgumentException("Illegal group reference");
				cursor++;

				// Capture the largest legal group string
				boolean done = false;
				while (!done) {
					if (cursor >= replacement.length()) {
						break;
					}
					int nextDigit = bytes[cursor] - '0';
					if ((nextDigit < 0)||(nextDigit > 9)) { // not a number
						break;
					}
					int newRefNum = (refNum * 10) + nextDigit;
					if (groupCount() < newRefNum) {
						done = true;
					} else {
						refNum = newRefNum;
						cursor++;
					}
				}
				sb.append(groupUTF8(refNum));
			} else {
				sb.append(nextByte);
				cursor++;
			}
		}

		prevEnd = end();
	}

	public void appendTail(StringUTF8Buffer sb) {
		if (inputUTF8.length() == prevEnd) {
			return;
		}
		sb.append(new StringUTF8(inputUTF8, prevEnd));
	}
}
