(ns exoscale.specs.string
  (:require [clojure.spec.alpha :as s]
            [clojure.spec.gen.alpha :as g]
            [clojure.string :as str]
            [exoscale.lingo :as lingo]
            [exoscale.lingo.impl :as lingo.impl]
            [net.cgrand.macrovich :as macros])
  #?(:cljs
     (:require-macros [net.cgrand.macrovich :as macros]
                      [exoscale.specs.string :refer [string-of]])))

(defn string-of*
  [s
   {:keys [min-length max-length length
           trim? blank? re]}]
  (and (string? s)
       (let [s (cond-> s trim? str/trim)
             len (count s)]
         (and
          (or (not (boolean? blank?))
              (= (str/blank? s) blank?))
          (or (not max-length)
              (<= len max-length))
          (or (not min-length)
              (>= len min-length))
          (or (not length)
              (= length len))
          (or (not re)
              (re-matches (re-pattern re) s))))))

(defn string-of-gen*
  [{:keys [max-length min-length length blank?]
    :or {min-length 0
         max-length 100}}]
  (let [min-length (if (and (false? blank?)
                            (zero? min-length))
                     1
                     min-length)]
    (g/fmap str/join
            (if length
              (g/vector (g/char-alphanumeric) length)
              (g/one-of
               [(g/vector (g/char-alphanumeric)
                          min-length
                          max-length)
                (g/return "")])))))

(macros/deftime
  (defmacro string-of
    "Returns a spec that will check conditions against string with
  AND. The options are loosely based on coll-of.

  * :trim? - trim value before checking

  Combined with AND:
  * :max-length - max size
  * :min-length - min size
  * :length - exact size
  * :re - either a regex or a string pattern
  * :blank? -check for blank"
    [opts]
    `(-> (s/and string? #(string-of* % ~opts))
         (s/with-gen #(string-of-gen* ~opts)))))

(lingo/set-pred-error!
 (s/cat :_ #{'exoscale.specs.string/string-of*}
        :_ #{'%}
        :opts map?)
 (fn [{:keys [opts]} _]
   (let [{:keys [length min-length max-length blank? re]} opts]
     (str "should be a String "
          (str/join ", "
                    (cond-> []
                      (false? blank?)
                      (conj "non blank")
                      min-length
                      (conj (lingo.impl/format "at least %d characters in length" min-length))
                      max-length
                      (conj (lingo.impl/format "at most %d characters in length" max-length))
                      length
                      (conj (lingo.impl/format "exactly %d characters in length" length))
                      re
                      (conj (lingo.impl/format "matching the regex %s" re))))))))
