(ns exoscale.specs.net
  (:require [clojure.spec.alpha :as s]
            [clojure.spec.gen.alpha :as gen]
            [clojure.string :as str])
  #?(:clj (:import (exoscale.specs.validator InetAddressValidator))
     :cljs (:import (goog.net ipaddress))))

#?(:clj (def ^:private ^InetAddressValidator validator
          (InetAddressValidator/getInstance)))

(s/def ::ipv4 (s/with-gen (s/and string?
                                 #?(:clj #(.isValidInet4Address validator %)
                                    :cljs #(try
                                             (new ipaddress.Ipv4Address %)
                                             (catch :default _))))
                #(gen/fmap (fn [x] (str/join "." x))
                           (gen/tuple (gen/return 159)
                                      (gen/large-integer* {:min 0 :max 255})
                                      (gen/large-integer* {:min 0 :max 255})
                                      (gen/large-integer* {:min 0 :max 255})))))

(s/def ::ipv6 (s/with-gen (s/and string?
                                 #?(:clj #(.isValidInet6Address validator %)
                                    :cljs #(try
                                             (new ipaddress.Ipv6Address %)
                                             (catch :default _))))
                #?(:clj (fn [] (gen/fmap (fn [x] (str/join ":" x))
                                        (gen/vector
                                         (gen/fmap #(format "%02x" %) (gen/large-integer* {:min 0 :max 65536}))
                                         8)))
                   :cljs #(s/gen #{"6845:ff8d:fb92:3cb3:d275:8d55:debc:e56c"
                                   "cf52:d613:98e7:d478:113:1008:abe5:9042"}))))

(s/def ::ip (s/or :ipv4 ::ipv4
                  :ipv6 ::ipv6))

(s/def ::mac-address
  (s/and string?
         #(re-matches #"([0-9a-fA-F]{2}:??){5}([0-9a-fA-F]{2})" %)))

(s/def ::port (s/int-in 0 65354))

(s/def ::url
  (s/and string?
         #(try
            (->> #?(:clj (java.net.URI. %)
                    :cljs (goog.Uri. %))
                 (.getScheme)
                 str/lower-case
                 (contains? #{"https" "http"}))
            (catch #?(:clj Exception :cljs :default) _))))

(defn- split-cidr
  [cidr]
  (let [parts (str/split cidr #"/")]
    (when (-> parts count (= 2))
      parts)))

(defn- valid-ipv4-ip-and-cidr?
  [[ip cidr]]
  (and (s/valid? ::ipv4 ip)
       (s/valid? #(when-let [block (parse-long %)]
                    (<= 0 block 32))
                 cidr)))

(defn- valid-ipv6-ip-and-cidr?
  [[ip cidr]]
  (and (s/valid? ::ipv6 ip)
       (s/valid? #(when-let [block (parse-long %)]
                    (<= 0 block 128))
                 cidr)))

(s/def ::cidr-ipv4
  (s/with-gen (s/and string?
                     #(->> %
                           (split-cidr)
                           (valid-ipv4-ip-and-cidr?)))
    #(gen/fmap (fn [x] (str (str/join "." x) "/32"))
               (gen/tuple (gen/return 159)
                          (gen/large-integer* {:min 0 :max 255})
                          (gen/large-integer* {:min 0 :max 255})
                          (gen/large-integer* {:min 0 :max 255})))))

(s/def ::cidr-ipv6
  (s/with-gen (s/and string?
                     #(->> %
                           (split-cidr)
                           (valid-ipv6-ip-and-cidr?)))
    #(s/gen #{"1:2:3:4:5:6:7:8/96" "::ffff:10.0.0.1/96" "::ffff:1.2.3.4/96"})))

(s/def ::cidr (s/or :cidr-ipv4 ::cidr-ipv4
                    :cidr-ipv6 ::cidr-ipv6))
