(ns exoscale.specs
  "Utilities extending clj.specs with missing bits (some of which will
  come with alpha2), like metadata, docstring support"
  (:refer-clojure :exclude [meta keys select-keys])
  (:require [clojure.spec.alpha :as s]
            [exoscale.coax.inspect :as si]
            [clojure.set :as set]))

;;; Metadata support
;;; The following only works only for registered specs

(defonce metadata-registry (atom {}))
(s/def ::metadata-registry-val (s/map-of qualified-keyword? any?))

(s/fdef vary-meta!
  :args (s/cat :k qualified-keyword?
               :f ifn?
               :args (s/* any?))
  :ret qualified-keyword?)
(defn vary-meta!
  "Like clojure.core/vary-meta but for registered specs, mutates the
  meta in place, return the keyword spec"
  [k f & args]
  (swap! metadata-registry
         #(update % k
                  (fn [m]
                    (apply f m args))))
  k)

(s/fdef with-meta!
  :args (s/cat :k qualified-keyword?
               :meta any?)
  :ret ::metadata-registry-val)
(defn with-meta!
  "Like clojure.core/with-meta but for registered specs, mutates the
  meta in place, return the keyword spec"
  [k m]
  (swap! metadata-registry
         #(assoc % k m))
  k)

(s/fdef meta
  :args (s/cat :k qualified-keyword?)
  :ret any?)
(defn meta
  "Like clojure.core/meta but for registered specs."
  [k]
  (get @metadata-registry k))

(s/fdef unregister-meta!
  :args (s/cat :k qualified-keyword?)
  :ret qualified-keyword?)
(defn unregister-meta!
  "Unregister meta data for a spec"
  [k]
  (swap! metadata-registry dissoc k)
  k)

(s/fdef with-doc
  :args (s/cat :k qualified-keyword?
               :doc string?)
  :ret qualified-keyword?)
(defn with-doc
  "Add doc metadata on a registered spec"
  [k doc]
  (vary-meta! k assoc :doc doc))

(s/fdef doc
  :args (s/cat :k qualified-keyword?)
  :ret (s/nilable string?))
(defn doc
  "Returns doc associated with spec"
  [k]
  (some-> k meta :doc))

(s/def ::def-args
  (s/cat :spec-key qualified-keyword?
         :meta (s/? map?)
         :docstring (s/? string?)
         :spec any?))

(s/fdef exoscale.specs/def
  :args ::def-args
  :ret any?)
(defmacro def
  "Same as `clojure.spec/def` but take an optional `docstring`, `attr-map`.

  ;; just a doctring:

  (exoscale.specs/def ::foo
    \"this is a foo\"
     string?)

  ;; add metadata
  (exoscale.specs/def ::foo
    \"this is a foo\"
    {:something :interesting}
    string?)

  It's also identical to `clojure.spec/def` with no options (as convenience):
  (exoscale.specs/def ::foo string?)

  It returns the spec key, just like `clojure.spec/def`"
  {:arglists '([spec-key spec]
               [spec-key doc-string spec]
               [spec-key metadata spec]
               [spec-key metadata doc-string spec])}
  [& args]
  (let [{:keys [spec-key docstring meta spec]} (s/conform ::def-args args)]
    `(do
       (s/def ~spec-key ~spec)
       ~@(cond-> []
           docstring
           (conj `(with-doc ~spec-key ~docstring))
           meta
           (conj `(vary-meta! ~spec-key merge ~meta)))
       ~spec-key)))

(defn ^:no-doc allowed-keys?
  [ks]
  #(set/superset? ks (set (clojure.core/keys %))))

(defn- un-ns
  [k]
  (keyword (name k)))

(defn- find-keys-flat
  [km]
  (filter keyword? (flatten km)))

(defn- find-keys
  [km]
  (set (concat (find-keys-flat (:req km))
               (map un-ns (find-keys-flat (:req-un km)))
               (:opt km)
               (map un-ns (:opt-un km)))))

(defn select-keys
  "Selects a set of keys from a map from it's spec. Works with s/keys,
  s/merge, s/multi-spec."
  [x spec]
  (letfn [(key-set [spec x]
            (let [[f & args :as _spec-form] (si/spec-root spec)]
              (condp = f
                `s/keys (find-keys (apply hash-map args))
                `s/merge (into #{}
                               (mapcat #(key-set % x))
                               args)
                `s/multi-spec (key-set (s/form ((resolve (first args)) x))
                                       x))))]
    (clojure.core/select-keys x (key-set spec x))))

(defmacro keys
  "Like clojure.spec.alpha/keys but allow to pass a `:closed?` parameter
  to make the spec strict on allowed keys."
  [& {:keys [closed?] :as args}]
  (let [keys-form `(s/keys ~@(apply concat args))]
    (if closed?
      `(-> (s/and ~keys-form
                  (allowed-keys? ~(find-keys args)))
           (s/with-gen #(s/gen ~keys-form)))
      keys-form)))
