(ns exoscale.checkmate.core-async
  (:require [clojure.core.async :as async]
            [exoscale.checkmate.impl :as impl]))

(defn- effects!
  [k {:as ctx :exoscale.checkmate/keys [conditions]}]
  (async/go
    (doseq [cd conditions]
      (when-let [f (get cd k)]
        (async/<! (f ctx)))
      conditions)))

(defn- attempt!
  [handler f ctx]
  (let [ctx (dissoc ctx :exoscale.checkmate/error)
        ch (async/promise-chan)]
    (async/take! (f ctx)
                 (fn [ctx-or-e]
                   (async/>!! ch
                              (if (instance? Exception ctx-or-e)
                                (impl/error-ctx ctx ctx-or-e)
                                (handler ctx ctx-or-e)))))
    ch))

(defn run
  ([f conditions] (run f conditions {}))
  ([f conditions opts]
   (impl/assert-conditions-valid! conditions)
   (let [{:as opts
          :exoscale.checkmate.hook/keys [success error failure]
          :exoscale.checkmate.core-async/keys [ch]
          :exoscale.checkmate/keys [handler return]}
         (merge impl/default-options opts)
         ch (or ch (async/promise-chan))
         ctx (-> (impl/init-ctx opts conditions)
                 (impl/setup-conditions :core-sync))]
     (effects! :exoscale.checkmate/setup-effect! ctx)
     (async/go-loop [ctx ctx]
       (when-let [ctx (async/<! (attempt! handler f ctx))]
         (if (impl/error? ctx)
           (let [ctx (impl/abort-ctx ctx)]
             (if (impl/fail? ctx)
               (do
                 (failure ctx)
                 (async/<! (effects! :exoscale.checkmate/failure-effect! ctx))
                 (async/>! ch
                           (-> (impl/teardown-conditions ctx)
                               :exoscale.checkmate/error)))
               (do
                 (error ctx)
                 (async/<! (effects! :exoscale.checkmate/error-effect! ctx))
                 (recur (impl/update-conditions ctx)))))
           (do
             (success ctx)
             (async/<! (effects! :exoscale.checkmate/success-effect! ctx))
             (async/>! ch (return (impl/teardown-conditions ctx)))))))
     ch)))
