(ns exoscale.checkmate
  "3 main states

  * success (all good)
  * error (bad, potentially retry)
  * failure (terminal failure)

  - Conditions control if we should retry and how to setup the ctx
  at first call, and how to update the ctx upon failures (inc retry
  counter for instance). Conditions are in effect, composable and
  should remain pure (ctx in/out).

  You can for instance do:

  ;; will retry up to 3 times with a 1000ms delay between tries
  (cm/do! #(do-something) [(delayed-retries (constant-backoff-delays 1000)) (max-retries 3)])

  - Effects can apply side effects at various stages (ex sleep, update
  state machine), they are purely for io, as opposed to Conditions"
  (:require [exoscale.checkmate.sync :as sync]
            [exoscale.ex :as ex]
            [clojure.core.async :as async]
            [manifold.time :as mt]
            [qbits.auspex :as a]
            [clojure.spec.alpha :as s])
  (:import (java.time Instant)))

(s/def :exoscale.checkmate/condition
  (s/keys :req [:exoscale.checkmate/id
                :exoscale.checkmate/retry?]
          :opt [:exoscale.checkmate/update
                :exoscale.checkmate/setup-effect!
                :exoscale.checkmate/error-effect!
                :exoscale.checkmate/failure-effect!
                :exoscale.checkmate/success-effect!]))

(s/def :exoscale.checkmate/id qualified-keyword?)
(s/def :exoscale.checkmate/retry? ifn?)
(s/def :exoscale.checkmate/update ifn?)
(s/def :exoscale.checkmate/setup-effect! ifn?)
(s/def :exoscale.checkmate/error-effect! ifn?)
(s/def :exoscale.checkmate/failure-effect! ifn?)
(s/def :exoscale.checkmate/success-effect! ifn?)

;;; Default runner
(defn run
  "Synchronous runner, take function `f` and runs it.
  Upon error will check Conditions passed as arguments to know if it
  warrants a retry, otherwise it triggers a failure and rethrows the
  last error.

  At every stage `hooks` are invoked, and potentially Effects from
  Conditions.

  * `hook`: map of allow to setup event at various stages (logging, reporting)
  * `conditions`: sequence of Conditions to be used"

  ([f conditions] (run f conditions {}))
  ([f conditions opts] (sync/run f conditions opts)))

;;; Conditions

(defn max-retries
  [max]
  #:exoscale.checkmate
  {:id ::max-retries
   :setup (fn [ctx] (assoc ctx :exoscale.checkmate/retries 0))
   :retry? (fn [ctx] (< (:exoscale.checkmate/retries ctx) max))
   :update (fn [ctx] (update ctx :exoscale.checkmate/retries inc))})

(defn retry-on*
  "like retry-on, but runs on the whole ctx instead of just result"
  [pred]
  #:exoscale.checkmate
  {:id ::retry-on*
   :retry? pred})

(defn retry-on
  "Allow retries on (pred ret) -> true, useful to limit retries to a
  class of errors ex: (retry-on #(instance? TimeoutException %)).  `x`
  can be a function, a Class name (ex: some exception type), or a
  keyword in that case it will match (exoscale.ex/type? x %)."
  [x]
  (let [pred (cond
               (ifn? x) x
               (instance? Class x) #(instance? x %)
               keyword? #(ex/type? x %))]
    (assoc (retry-on* #(pred (:exoscale.checkmate/error %)))
           :exoscale.checkmate/id ::retry-on)))

(defn fail-on
  "Set to failed on (pred ret) -> true Similar to retry-on, the other
  way around. Fails on match."
  [pred]
  (assoc (retry-on (complement pred))
         :exoscale.checkmate/id ::fail-on))

(defn delays
  [coll]
  #:exoscale.checkmate
  {:id ::delays
   :setup (fn [ctx] (assoc ctx :exoscale.checkmate/delays coll))
   :retry? (fn [ctx] (seq (:exoscale.checkmate/delays ctx)))
   :update (fn [ctx] (update ctx :exoscale.checkmate/delays rest))
   :error-effect! (fn [ctx]
                    (when-let [delay (some-> ctx :exoscale.checkmate/delays first)]
                      ;; we could open this, but meh, with the async proto we were
                      ;; talking about maybe
                      (case (:exoscale.checkmate/runner ctx)
                        :sync (Thread/sleep delay)
                        :core-async (async/timeout delay)
                        :manifold (mt/in delay (constantly ::noop))
                        :auspex (a/timeout! (a/future) delay ::noop))))})

(defn timeout
  ([ms] (timeout ms {}))
  ([ms {:as _opts :keys [exception]}]
   #:exoscale.checkmate
   {:id ::timeout
    :setup (fn [ctx]
             (assoc ctx :exoscale.checkmate.timeout/inst (.plusMillis (Instant/now) ms)))
    :retry? (fn [ctx]
              (.isBefore (Instant/now)
                         (:exoscale.checkmate.timeout/inst ctx)))
    :failure-effect! (fn [ctx]
                      (when (and (ifn? exception)
                                 (-> ctx :exoscale.checkmate/abort-cause ::id (= ::timeout)))
                        (exception ctx)))}))

;;; following are to be used as arguments to delayed-retries
(defn constant-backoff-delays [ms]
  (repeat ms))

(defn exponential-backoff-delays
  [x]
  (iterate #(long (* Math/E %)) x))

(defn progressive-backoff-delays [x]
  (lazy-cat
   (repeat x 100)
   (repeat x 500)
   (repeat x 1500)
   (repeat x 15000)
   (repeat 60000)))

(defn rate-limited-retries
  "!! This condition should be shared for all runner calls, since it's
  stateful

  Only allows a max rate of errors until it plain rejects calls until
  rate can be satisfied again:

  Example:

  max-errors : 10
  error-rate : 1000ms

  Allows up to 10 successive errors, upon token exhaustion will only
  allow 1 error per second in burst, until it fills again to 10.

  So this allows burst'iness up to a point.

  If you create a lot or rate-limited-retries conditions you need to
  remember to async/close! the associated chan, or pass it as
  argument. The chan is also available via Condition ctx, in case
  you want to enable temporary burst'iness, early flush or closing"
  [{:keys [max-errors error-rate exception]
    :or {max-errors 10
         error-rate 1000
         exception (fn [ctx]
                     (ex-info "Rejected execution, max rate reached"
                              {:type :exoscale.ex/busy
                               :exoscale.checkmate/ctx ctx}))}}]

  (let [ch (async/chan max-errors)
        reject? (atom false)]
    ;; we will create a token emitting chan at desired
    ;; rate, upon errors call we try to poll! from that
    ;; chan, if it fails we know rate is exceeded and we
    ;; can just reject.
    (dotimes [_ max-errors]
      (async/>!! ch :exoscale.checkmate/token))

    (async/go
      ;; then feed at rate
      (loop []
        (async/<! (async/timeout error-rate))
        (when (async/>! ch :exoscale.checkmate/token)
          (reset! reject? false)
          (recur))))
    #:exoscale.checkmate
    {:id ::rate-limiter
     :setup (fn [ctx] (assoc ctx :exoscale.checkmate/ch ch))
     :retry? (fn [_ctx]
               ;; no token available, no retry
               (async/poll! ch))
     :setup-effect! (fn [ctx]
                      ;; do not allow calls while it's rate limited
                      (when @reject?
                        (throw (exception ctx))))
     :failure-effect! (fn [ctx]
                        ;; stop allowing request until next token tick
                        (reset! reject? true)
                        (throw (exception ctx)))}))
