# checkmate

[![Clojars Project](https://img.shields.io/clojars/v/exoscale/checkmate.svg)](https://clojars.org/exoscale/checkmate) [![cljdoc badge](https://cljdoc.xyz/badge/exoscale/checkmate)](https://cljdoc.org/d/exoscale/checkmate/CURRENT/api/exoscale.checkmate)


> By failing to prepare, you are preparing to fail


A library to handle failure/retries/rate-limiting.

``` clj
(require 'exoscale.checkmate :refer :all)

;; runs at most 3 times
(run #(...) [(max-retries 3)])

;; at most 3 times with delays
(run #(...) [(delays (repeat 3 1000))])

;; they compose, infinite seq of 1s pause with max-retries 3
(run #(...) [(delays (repeat 1000)) (max-retries 3)])

;; without max-retries it's essentially a "wait-for-it", forever repeating calls
;; until success thing. Not sure it's a good idea, but it's possible
(run #(...) [(delays (repeat 1000))])

;; fail or retry only in matched cases
(run #(...) [(retry-on TimeoutException) (max-retries 3))])

(run #(...) [(fail-on ConnectionException) (max-retries 3))])

;; works with exoscale.ex (matches ex-data :type)
(run #(...) [(retry-on :exoscale.ex/busy) (max-retries 3))])

;; works with simple predicates
(run #(...) [(retry-on #(...)) (max-retries 3)])

;; everything composes

(run #(...)
     [(retry-on TimeoutException)
      (fail-on ConnectionException)
      (max-retries 3)])

;; there is also retry-on* will allow to match anything on the ctx

;; generating delays is easy, they are just seqs:
;; snippet from the condition ns:

(defn constant-backoff-delays [ms]
  (repeat ms))

(defn exponential-backoff-delays [x]
  (iterate #(* Math/E %) x))

(defn progressive-backoff-delays [x]
  (lazy-cat
   (repeat x 100)
   (repeat x 500)
   (repeat x 1500)
   (repeat x 15000)
   (repeat 60000)))

[...]

;;; More complex (potentially stateful across calls) condition, Rate limiter:
;; just run at most 3 times, use shared rate-limiter with initial token size of
;; 10 and allowing only up to 1 retry run per sec after error token exhaustion

(def rl (rate-limiter {:max-errors 10 :error-rate 1000}))

;; these 3 calls share the rate limiter

(run #(...) [rl (max-retries 3)])

(run #(...) [rl (max-retries 10)])

(run #(...) [rl (max-retries 5)])

;; the same approach can used be implement count/time based circuit breakers

;; hooks (error, failure, success)
(run #(...)
     [...]
     {:exoscale.checkmate.hook/error (fn [err] (log/error "boom, retrying" err))   ... })

;; manifold, returns a deferred
(require '[exoscale.checkmate.manifold :as mm])
(mm/run #(something-returning-a-deferred) [...])

;; core.async, returns a promise-chan
(require '[exoscale.checkmate.core-async :as ca])
(ca/run #(something-returning-a-chan) [...])

;; CompletableFuture via auspex, returns a completionstage
(require '[exoscale.checkmate.auspex :as cx])
(cx/run #(something-returning-a-completablefuture) [...])
```

checkmate is made of simple building blocks:

* Runners (one for sync and others for core.async, manifold, CompletableFuture)
* Conditions
* Effects

Conditions control if we should retry and how to setup the ctx at
first call, and how to update the ctx upon failures (inc retry counter
for instance). Conditions are in fact, **composable** and should
remain pure (ctx in/out). Conditions implementations should be/are
compatible with any runner since they are pure.

Effects can apply side effects at various stages (ex sleep, update
state machine), they are purely for io, as opposed to Conditions.
Effects compatibility with any runner depends on what they do, either
we branch internally depending on the runner used or we can create
implementations of Effects per type of runner when necessary.

The simplest condition would be for max-retries:

``` clj
(defn max-retries
  [max]
  (reify Condition
    (setup! [this ctx]
      (assoc ctx :exoscale.checkmate/retries 0))
    (retry? [this ctx]
      (< (:exoscale.checkmate/retries ctx) max))
    (update! [this ctx]
      (update ctx :exoscale.checkmate/retries inc))))
```

setups ctx counter per `run` to 0, a check to know if we need to
retry from the ctx and an `update!` to refresh ctx after an error


A slightly more involved condition with an effect (simplified version):

``` clj
(defn delays
  [delays]
  (reify
    Condition
    (setup! [this ctx]
      (assoc ctx
             :exoscale.checkmate/delays delays))
    (retry? [this ctx]
      (seq (:exoscale.checkmate/delays ctx)))
    (update! [this ctx]
      (update ctx :exoscale.checkmate/delays rest))
    ErrorEffect
    (error-effect! [this ctx]
       (Thread/sleep (-> ctx :exoscale.checkmate/delays first)))))
```

Effects do not affect ctx, but they can affect execution, for
instance apply a pause or cause an execution rejection in the case of
the rate limiter. They should not be used for logging as they are tied
to a condition's implementation, hooks are better for this since they
are per call.

## Documentation

[![cljdoc badge](https://cljdoc.xyz/badge/exoscale/checkmate)](https://cljdoc.org/d/exoscale/checkmate/CURRENT/api/exoscale.checkmate)

## Installation

checkmate is [available on Clojars](https://clojars.org/exoscale/checkmate).

Add this to your dependencies:

[![Clojars Project](https://img.shields.io/clojars/v/exoscale/checkmate.svg)](https://clojars.org/exoscale/checkmate)

## License

Copyright © 2020 [Exoscale](https://exoscale.com)
