(ns exoscale.checkmate
  "3 main states

  * success (all good)
  * error (bad, potentially retry)
  * failure (terminal failure)

  - Conditions control if we should retry and how to setup the ctx
  at first call, and how to update the ctx upon failures (inc retry
  counter for instance). Conditions are in effect, composable and
  should remain pure (ctx in/out).

  You can for instance do:

  ;; will retry up to 3 times with a 1000ms delay between tries
  (cm/do! #(do-something) [(delayed-retries (constant-backoff-delays 1000)) (max-retries 3)])

  - Effects can apply side effects at various stages (ex sleep, update
  state machine), they are purely for io, as opposed to Conditions"
  (:require [exoscale.checkmate.protocols :as p]
            [exoscale.checkmate.sync :as sync]
            [exoscale.ex :as ex]
            [clojure.core.async :as async]
            [manifold.time :as mt]
            [qbits.auspex :as a]))


;;; Default runner


(defn run
  "Synchronous runner, take function `f` and runs it.
  Upon error will check Conditions passed as arguments to know if it
  warrants a retry, otherwise it triggers a failure and rethrows the
  last error.

  At every stage `hooks` are invoked, and potentially Effects from
  Conditions.

  * `hook`: map of allow to setup event at various stages (logging, reporting)
  * `conditions`: sequence of Conditions to be used"

  ([f conditions] (run f conditions {}))
  ([f conditions opts] (sync/run f conditions opts)))

;;; Conditions

(defn max-retries
  [max]
  (reify p/Condition
    (setup! [this ctx]
      (assoc ctx :exoscale.checkmate/retries 0))
    (retry? [this ctx]
      (< (:exoscale.checkmate/retries ctx) max))
    (update! [this ctx]
      (update ctx :exoscale.checkmate/retries inc))))

(defn retry-on*
  "like retry-on, but runs on the whole ctx instead of just result"
  [pred]
  (reify p/Condition
    (setup! [this ctx] ctx)
    (retry? [this ctx] (pred ctx))
    (update! [this ctx] ctx)))

(defn retry-on
  "Allow retries on (pred ret) -> true, useful to limit retries to a
  class of errors ex: (retry-on #(instance? TimeoutException %)).  `x`
  can be a function, a Class name (ex: some exception type), or a
  keyword in that case it will match (exoscale.ex/type? x %)."
  [x]
  (let [pred (cond
               (ifn? x) x
               (instance? Class x) #(instance? x %)
               keyword? #(ex/type? x %))]
    (retry-on* #(pred (:exoscale.checkmate/result %)))))

(defn fail-on
  "Set to failed on (pred ret) -> true Similar to retry-on, the other
  way around. Fails on match."
  [pred]
  (retry-on (complement pred)))

(defn delays
  [coll]
  (reify
    p/Condition
    (setup! [this ctx]
      (assoc ctx
             :exoscale.checkmate/delays coll))
    (retry? [this ctx]
      (seq (:exoscale.checkmate/delays ctx)))
    (update! [this ctx]
      (update ctx :exoscale.checkmate/delays rest))
    p/ErrorEffect
    (error-effect! [this ctx]
      (when-let [delay (some-> ctx :exoscale.checkmate/delays first)]
        ;; we could open this, but meh, with the async proto we were
        ;; talking about maybe
        (case (:exoscale.checkmate/runner ctx)
          :sync (Thread/sleep delay)
          :core-async (async/timeout delay)
          :manifold (mt/in delay (constantly ::noop))
          :auspex (a/timeout! (a/future) delay ::noop))))))

;;; following are to be used as aruments to delayed-retries
(defn constant-backoff-delays [ms]
  (repeat ms))

(defn exponential-backoff-delays [x]
  (iterate #(* Math/E %) x))

(defn progressive-backoff-delays [x]
  (lazy-cat
   (repeat x 100)
   (repeat x 500)
   (repeat x 1500)
   (repeat x 15000)
   (repeat 60000)))

(defn rate-limited-retries
  "!! This condition should be shared for all runner calls, since it's
  ctxful

  Only allows a max rate of errors until it plain rejects calls until
  rate can be satisfied again:

  Example:

  max-errors : 10
  error-rate : 1000ms

  Allows up to 10 successive errors, upon token exhaustion will only
  allow 1 error per second in burst, until it fills again to 10.

  So this allows burst'iness up to a point.

  If you create a lot or rate-limited-retries conditions you need to
  remember to async/close! the associated chan, or pass it as
  argument. The chan is also available via Condition ctx, in case
  you want to enable temporary burst'iness, early flush or closing"
  [{:keys [max-errors error-rate exception]
    :or {max-errors 10
         error-rate 1000
         exception (fn [condition ctx]
                     (ex-info "Rejected execution, max rate reached"
                              {:type :exoscale.ex/busy
                               :exoscale.checkmate/condition condition :exoscale.checkmate/ctx ctx}))}}]

  (let [ch (async/chan max-errors)
        reject? (atom false)]
    ;; we will create a token emitting chan at desired
    ;; rate, upon errors call we try to poll! from that
    ;; chan, if it fails we know rate is exceeded and we
    ;; can just reject.
    (dotimes [_ max-errors]
      (async/>!! ch :exoscale.checkmate/token))

    (async/go
      ;; then feed at rate
      (loop []
        (async/<! (async/timeout error-rate))
        (when (async/>! ch :exoscale.checkmate/token)
          (reset! reject? false)
          (recur))))

    (reify p/Condition
      (setup! [this ctx]
        ;; to allow the user to potentially close/feed it
        (assoc ctx :exoscale.checkmate/ch ch))
      (update! [this ctx] ctx)
      (retry? [this ctx]
        ;; no token available, no retry
        (async/poll! ch))
      p/SetupEffect
      (setup-effect! [this ctx]
        ;; do not allow calls while it's rate limited
        (when @reject?
          (throw (exception this ctx))))

      p/FailureEffect
      (failure-effect! [this ctx]
        ;; stop allowing request until next token tick
        (reset! reject? true)
        (throw (exception this ctx))))))
