(ns exoscale.checkmate.sync
  "Retry library.

  3 main states

  * success (all good)
  * error (bad, potentially retry)
  * failure (terminal failure)

  - Conditions control if we should retry and how to setup the state
  at first call, and how to update the state upon failures (inc retry
  counter for instance). Conditions are in effect, composable and
  should remain pure (state in/out).

  You can for instance do:

  ;; will retry up to 3 times with a 1000ms delay between tries
  (cm/do! #(do-something) [(delayed-retries (constant-backoff-delays 1000)) (max-retries 3)])

  - Effects can apply side effects at various stages (ex sleep, update
  state machine), they are purely for io, as opposed to Conditions"
  (:require [exoscale.checkmate.condition :as c]
            [exoscale.checkmate.impl :as impl]))

(defn ^:no-doc attempt!
  "because loop will not allow recur from catch"
  [f]
  (try
    [:exoscale.checkmate/success (f)]
    (catch Exception e
      [:exoscale.checkmate/error e])
    (catch AssertionError e
      [:exoscale.checkmate/error e])))

(defn ^:no-doc effects!
  [proto f conditions state]
  (run! (fn [cd]
          (when (satisfies? proto cd)
            (f cd state)))
        conditions))

(def ^:no-doc setup-effects! (partial effects! c/SetupEffect #'c/setup-effect!))
(def ^:no-doc error-effects! (partial effects! c/ErrorEffect #'c/error-effect!))
(def ^:no-doc failure-effects! (partial effects! c/FailureEffect #'c/failure-effect!))
(def ^:no-doc success-effects! (partial effects! c/SuccessEffect #'c/success-effect!))

(defn run
  "Synchronous runner, take function `f` and runs it.
  Upon error will check Conditions passed as arguments to know if it
  warrants a retry, otherwise it triggers a failure and rethrows the
  last error.

  At every stage `hooks` are invoked, and potentially Effects from
  Conditions.

  * `hooks`: map of allow to setup event at various stages (logging, reporting)
  * `conditions`: sequence of Conditions to be used"
  ([f conditions] (run f conditions {}))
  ([f conditions opts]
   (let [{:as opts
          :exoscale.checkmate.hooks/keys [success error failure]}
         (merge impl/default-options opts)
         state (impl/setup-conditions! conditions opts :sync)]
     (setup-effects! conditions state)
     (loop [state state]
       (let [[status ret] (attempt! f)
             state (assoc state :exoscale.checkmate/result ret)]
         (case status
           :exoscale.checkmate/success
           (do
             (success ret)
             (success-effects! conditions state)
             ret)
           :exoscale.checkmate/error
           ;; check continue on all conditions, abort on first false
           (if (impl/retry? conditions state)
             (do
               (error ret)
               (error-effects! conditions state)
               (recur (impl/update-conditions! conditions state)))
             (do
               (failure ret)
               (failure-effects! conditions state)
               (throw ret)))))))))
