(ns blueprint.server
  "An encoding of the most common API building case, allows
   building servers with a map of API command to handling
   function, leaving wiring to this"
  (:require [aleph.http            :as http]
            [aleph.netty           :as netty]
            [blueprint.handler     :as handler]
            [clojure.spec.alpha    :as s]
            [clojure.tools.logging :as log]
            [exoscale.interceptor.manifold :as ixm]
            [exoscale.ex           :as ex]))

(def command-map
  #'handler/command-map)

(defn start
  "Start a server, expects the output of `build`,
   returns a modified map with an additional `:aleph.http/server`
   key which holds the Aleph server."
  [this]
  (handler/start this))

(defn stop
  "Stop a server, expects the output of `build`. Returns
   the input map without the `::http/server` key."
  [this]
  (handler/stop this))

(defn- start-callback
  [{::http/keys      [port options]
    :as              this}]
  (ex/assert-spec-valid ::handler/build-server this)
  (let [options      (assoc options :port (or port (:port options) 8080))]
    (log/infof "Starting server on %s port %s"
               (or (:host options) "localhost")
               (:port options))
    (assoc this ::http/server (http/start-server (handler/ring-handler this) options))))

(defn- stop-callback
  [{::http/keys [server] :as http-server}]
  (log/infof "Stopping server")
  (when (some? server)
    (.close ^java.io.Closeable server)
    (netty/wait-for-close server))
  (dissoc http-server ::http/server))

(defn- translate-keys
  "For backward compabitility. Any use of ServerDispatcher will have to
   be adapted though."
  [{::keys [dispatchers dispatch-map commands] :as this}]
  (cond-> (dissoc this ::dispatchers ::dispatch-map ::commands)
    (some? dispatchers)
    (assoc ::handler/dispatchers dispatchers)
    (some? dispatch-map)
    (assoc ::handler/dispatch-map dispatchers)
    (some? commands)
    (assoc ::handler/commands commands)))

(defn build
  "Builds a map describing a stopped API server,
   ready for use by `start` and `stop`.

  Acts as a component if wired to a system map.

  No required checking of configuration yet, this is deferred
  to `start` in case dynamic wiring of dependencies
  is done."
  ([]
   (build {}))
  ([this]
   (handler/build (translate-keys this)
                  start-callback
                  stop-callback
                  ixm/execute)))

(s/def ::http/port (partial s/int-in-range? 0 65536))
(s/def ::http/options map?)
