(ns blueprint.server
  "An encoding of the most common API building case, allows
   building servers with a map of API command to handling
   function, leaving wiring to this"
  (:require [aleph.http            :as http]
            [aleph.netty           :as netty]
            [blueprint.core        :as blueprint]
            [blueprint.handler     :as handler]
            [blueprint.openapi     :as openapi]
            [blueprint.router      :as router]
            [clojure.spec.alpha    :as s]
            [clojure.tools.logging :as log]
            [exoscale.ex           :as ex]))

(defprotocol BlueprintDispatcher
  :extend-via-metadata true
  (command-map [this] "Yield a partial dispatch map"))

(defn api-handler-fn
  "The default handler function, at the end of a blueprint
   interceptor chain. When reached, a valid command has been
   parsed.

   A provided dispatch map is looked up for the corresponding
   handler function and hands off to it."
  [definition dispatch-map]
  (let [known-handlers (-> definition :commands keys set)]
    (fn [{:keys [handler] :as request}]
      (let [handler-fn (get dispatch-map handler)]
        (cond
          (some? handler-fn)
          (handler-fn request)

          (contains? known-handlers handler)
          {:status 501
           :body   {:message (str "command " (name handler) " is missing")}}

          :else
          {:status 404
           :body   {:message "not found"}})))))

(defn find-dispatcher
  "Resolve key to a dispatcher. Prefers protocol implemenations,
   namespaced keys, or yields the value"
  [this k]
  (when-let [dispatcher (get this k)]
    (cond
      (satisfies? BlueprintDispatcher dispatcher) (command-map dispatcher)
      (contains? (meta dispatcher) `command-map)  (command-map dispatcher)
      (some? (::commands dispatcher))             (::commands dispatcher)
      :else                                       dispatcher)))

(defn build-dispatch-map
  "Walk the collection of dispatcher paths, representing keys
   into the input map that should be looked up to find partial
   dispatch maps.

   Elements found may implement BlueprintDispatcher or contain
   a `:blueprint.server/commands` key. When neither of these
   are true, the map is returned as is.

   All provided dispatchers are merged together and should be
   a valid dispatch-map implementation (a map of keyword to
   values implementing `clojure.lang.IFn`."
  [{::keys [dispatch-map dispatchers] :as this}]
  (ex/assert-spec-valid
   ::dispatch-map
   (if (some? dispatch-map)
     dispatch-map
     (into {} (map (partial find-dispatcher this) dispatchers)))))

(defn start
  "Start a server, expects the output of `build`,
   returns a modified map with an additional `:aleph.http/server`
   key which holds the Aleph server."
  [{::blueprint/keys [definition]
    ::handler/keys   [handler]
    ::keys           [dispatchers dispatch-map]
    ::http/keys      [port options]
    :as              this}]
  (ex/assert-spec-valid ::http-server this)
  (let [handler      (or handler
                         (api-handler-fn definition (build-dispatch-map this)))
        options      (assoc options :port (or port (:port options) 8080))
        ring-handler (-> definition
                         blueprint/parse
                         openapi/generate-openapi
                         router/generate-router
                         (handler/ring-handler handler this))]
    (log/infof "Starting server on %s port %s"
               (or (:host options) "localhost")
               (:port options))
    (assoc this ::http/server (http/start-server ring-handler options))))

(defn stop
  "Stop a server, expects the output of `build`. Returns
   the input map without the `::http/server` key."
  [{::http/keys [server] :as http-server}]
  (log/infof "Stopping server")
  (when (some? server)
    (.close ^java.io.Closeable server)
    (netty/wait-for-close server))
  (dissoc http-server ::http/server))

(defn build
  "Builds a map describing a stopped API server,
   ready for use by `start` and `stop`.

  Acts as a component if wired to a system map.

  No required checking of configuration yet, this is deferred
  to `start` in case dynamic wiring of dependencies
  is done."
  ([]
   (build {}))
  ([this]
   (ex/assert-spec-valid ::build-server this)
   (with-meta this
     {'com.stuartsierra.component/start start
      'com.stuartsierra.component/stop  stop})))

(s/def ::dispatch-map (s/map-of keyword? ifn?))
(s/def ::dispatchers (s/coll-of keyword?))

(s/def ::http/port (partial s/int-in-range? 0 65536))
(s/def ::http/options map?)
(s/def ::http-server
  (s/keys :req [::blueprint/definition
                (or ::handler/handler ::dispatchers ::dispatch-map)]
          :opt [::http/port ::http/options]))
(s/def ::build-server (s/keys :opt [::blueprint/definition
                                    ::handler/handler
                                    ::dispatch-map
                                    ::dispatchers
                                    ::http/port
                                    ::http/options]))
