(ns blueprint.router
  (:require [reitit.core :as r]
            [exoscale.ex :as ex]))

(defn gen-path
  [elems]
  (reduce
   str
   (for [[_ v] elems]
     (if (string? v) v (format "{%s}" (name (:name v)))))))

(defn prepare-route
  [options routers opname {:keys [path]}]
  (let [path-prefix (get options :path-prefix "")
        {:keys [method elems]} path]
    (update routers method conj [(str path-prefix (gen-path elems)) opname])))

(defn build-router
  [api-def routers]
  (fn [{:keys [request-method uri] :as request}]
    (let [{:keys [data path-params]} (r/match-by-path (get routers request-method) uri)]
      (cond-> request
        (some? data)
        (assoc :handler (:name data)
               ::options (get-in api-def [:commands (:name data) :options] {}))

        (nil? data)
        (assoc :handler :blueprint.core/not-found)

        (some? path-params)
        (assoc :path-params path-params)))))

(defn generate-router
  "Takes an api definition that has been parsed by `blueprint.core/parse`"
  ([api-def]
   (generate-router api-def {}))
  ([{:keys [commands router] :as api-def} options]
   (assoc api-def
          :router
          (build-router api-def
                        (->> (reduce-kv (partial prepare-route (:blueprint.options api-def)) {} commands)
                             (reduce-kv #(assoc %1 %2 (r/router %3 (merge {:syntax #{:bracket}}
                                                                          options))) {})))

          :assets
          (:assets router))))

(defn build-url
  [elems params]
  (for [e elems]
    (if (string? e) e (str (get params (first e))))))

(defn uri-for
  [{:keys [commands]} nick params]
  (if-let [{[method & elems] :path} (get commands nick)]
    {:request-method method
     :url            (reduce str "" (build-url elems params))}
    (throw (ex-info "unknown command" {:type ::ex/not-found}))))
