(ns blueprint.openapi
  "Generates the OAS spec from the blueprint definition.
  Reference docs: https://swagger.io/specification/

  OAS keys are already camelCased."
  (:require [blueprint.registry :as reg]
            [exoscale.specs :as spec]
            [clojure.set :as set]
            [clojure.string :as str]
            [clojure.walk :as walk]
            [jsonista.core :as json])
  (:import [clojure.lang Named]))

(defn build-ref
  [resource]
  {:$ref (format "#/components/schemas/%s" (name resource))})

(defmulti build-pred  :pred)
(defmethod build-pred 'any?     [_] {:type "object"})
(defmethod build-pred 'uuid?    [_] {:type "string" :format "uuid"})
(defmethod build-pred 'string?  [_] {:type "string"})
(defmethod build-pred 'keyword?  [_] {:type "string"})
(defmethod build-pred 'boolean? [_] {:type "boolean"})
(defmethod build-pred 'inst?    [_] {:type "string" :format "date-time"})
(defmethod build-pred 'int? [_] {:type "integer" :format "int64"})
(defmethod build-pred 'pos-int? [_] {:type             "integer"
                                     :format           "int64"
                                     :minimum          0
                                     :exclusiveMinimum true})
(defmethod build-pred 'nat-int? [_] {:type             "integer"
                                     :format           "int64"
                                     :minimum          0
                                     :exclusiveMinimum false})
(defmethod build-pred 'number? [_] {:type "number"})

(defmulti build-schema :type)

(defmethod build-schema :pred
  [input]
  (build-pred input))

(defmethod build-schema :>
  [{:keys [floor ceiling]}]
  (if (some? ceiling)
    {:type "integer"
     :format "int64"
     :minimum ceiling
     :maximum floor
     :exclusiveMinimum true
     :exclusiveMaximum true}
    {:type "integer"
     :format "int64"
     :minimum floor
     :exclusiveMinimum true}))

(defmethod build-schema :>=
  [{:keys [floor ceiling]}]
  (if (some? ceiling)
    {:type "integer"
     :format "int64"
     :minimum ceiling
     :maximum floor
     :exclusiveMinimum false
     :exclusiveMaximum false}
    {:type "integer"
     :format "int64"
     :minimum floor
     :exclusiveMinimum false}))

(defmethod build-schema :<=
  [{:keys [floor ceiling]}]
  (if (some? ceiling)
    {:type "integer"
     :format "int64"
     :minimum floor
     :maximum ceiling
     :exclusiveMinimum false
     :exclusiveMaximum false}
    {:type "integer"
     :format "int64"
     :maximum floor
     :exclusiveMaximum false}))

(defmethod build-schema :<
  [{:keys [floor ceiling]}]
  (if (some? ceiling)
    {:type "integer"
     :format "int64"
     :minimum floor
     :maximum ceiling
     :exclusiveMinimum true
     :exclusiveMaximum true}
    {:type "integer"
     :format "int64"
     :maximum floor
     :exclusiveMaximum true}))

(defmethod build-schema :not=
  [{:keys [val]}]
  {:not
   {:type "integer"
    :format "int64"
    :minimum val
    :maximum val
    :exclusiveMinimum true
    :exclusiveMaximum true}})

(defmethod build-schema :enum
  [{:keys [values]}]
  (if (empty? values)
    {:type "string" :enum (map name values)}
    ;;else
    ;; https://swagger.io/docs/specification/data-models/data-types/
    (cond (or (every? #(instance? Named %) values)
              (every? #(instance? CharSequence %) values))
          {:type "string" :enum (doall (map name values))}

          (every? #(instance? Number %) values)
          {:type "number" :enum (into [] values)}

          :else
          {:type "object" :enum (into [] values)})))

(defmethod build-schema :ip
  [_]
  {:type "string"
   :format "ip"})

(defmethod build-schema :ipv4
  [_]
  {:type "string"
   :format "ipv4"})

(defmethod build-schema :ipv6
  [_]
  {:type "string"
   :format "ipv6"})

(defmethod build-schema :reference
  [{:keys [reference]}]
  (build-ref reference))

(defmethod build-schema :spec
  [{:keys [spec] :as input}]
  (if-let [openapi (some-> spec spec/meta :blueprint.openapi/schema)]
    openapi
    {:type "unknown" :input input}))

(defmethod build-schema :map-of
  [{:keys [val-def]}]
  {:type "object" :additionalProperties (build-schema val-def)})

(defmethod build-schema :coll-of
  [{:keys [def opts]}]
  (let [{:keys [distinct max-count min-count count]} opts]
    (cond-> {:type "array" :items (build-schema def)}
      distinct
      (assoc :uniqueItems distinct)

      min-count
      (assoc :minItems min-count)

      max-count
      (assoc :maxItems max-count)

      count
      (assoc :minItems count
             :max-Items count))))

(defmethod build-schema :nilable
  [{:keys [def _]}]
  (assoc (build-schema def) :nullable true))

(defmethod build-schema :set-of
  [{:keys [def]}]
  {:type "array"
   :items (build-schema def)
   :uniqueItems true})

(defmethod build-schema :string-of
  [{{:keys [max-length min-length blank? length re]} :opts}]
  (let [min-length (or length min-length)
        max-length (or length max-length)]
    (cond-> {:type "string"}
      max-length (assoc :maxLength max-length)
      (false? blank?) (assoc :minLength 1)
      min-length (assoc :minLength min-length)
      re (assoc :pattern re))))

(defn build-attr
  [{:keys [def opts]}]
  (cond-> (build-schema def)
    (:ro? opts)             (assoc :readOnly true)
    (:wo? opts)             (assoc :writeOnly true)

    (some? (:desc opts))    (assoc :description (:desc opts))
    (some? (:example opts)) (assoc :example (:example opts))))

(defmethod build-schema :map
  [{:keys [attributes opts]}]
  (cond->
      {:type       "object"
       :required  (->> attributes
                       (filter (comp :req? :opts))
                       (map :attribute)
                       (map name))
       :properties (reduce #(assoc %1 (:attribute %2) (build-attr %2))
                           {}
                           attributes)}
    (some? (:desc opts))
    (assoc :description (:desc opts))
    (some? (:example opts))
    (assoc :example (:example opts))))

(defmethod build-schema :default
  [input]
  {:type "unknown" :input input})

(defn build-arg
  [{:keys [name def]}]
  {:in       "path"
   :required true
   :name     (clojure.core/name name)
   :schema   (build-schema def)})

(defn build-param
  [[name def]]
  {:in       "query"
   :required false
   :name     (clojure.core/name name)
   :schema   (build-schema def)})

(defn build-endpoint
  [elems]
  (->>
   (for [[t e] elems]
     (if (= :string t)
       e
       (format "{%s}" (name (:name e)))))
   (reduce str)))

(defn- prepare-extensions
  "Prefixes all the keys in the `extensions` map with `x-`"
  [extensions]
  (reduce-kv (fn [m k v]
               (assoc m (keyword (str "x-" (name k)))
                      v))
             {}
             ;;stringify will prevent :keyword to camelCase transformation
             ;;when spitting the json
             (walk/stringify-keys extensions)))

(defn- prefix-extensions
  "Extracts `ext-key` extensions out of `m` and prepends
  the openapi `x-` prefix to each extension"
  [m ext-key]
  (let [extensions (get m ext-key)]
    (merge (dissoc m ext-key)
           (prepare-extensions extensions))))

(defn build-output
  [code def output-extensions]
  (let [code-extensions (get output-extensions code)
        prefixed-exts   (prepare-extensions code-extensions)]
    (-> {:description (str code)
         :content     {"application/json" {:schema (build-schema def)}}}
        (merge prefixed-exts))))

(defn filter-args
  [elems]
  (for [[t e] elems :when (= :arg t)] e))

(defn generate-path
  [{:keys [tags input output params path output-extensions] :as cmd}]
  (let  [{:keys [method elems]} path
         args                   (filter-args elems)
         endpoint               (build-endpoint elems)]
    (cond->
        {:method       (name method)
         :path         endpoint
         :tags         (mapv name tags)
         :responses    (reduce-kv #(assoc %1 %2
                                          (build-output %2 %3 output-extensions))
                                  {} output)
         :description  (:desc cmd)
         :parameters   (concat (mapv build-arg args)
                               (mapv build-param params))
         :summary      (or (:summary cmd) "")}
      (some? input)
      (assoc :requestBody
             {:required true
              :content  {"application/json" {:schema (build-schema input)}}}))))

(defn generate-schemas
  [resources]
  (reduce-kv #(assoc %1 (name %2) (build-schema %3))
             {}
             resources))

(defn operation-name
  [op]
  (let [ns (namespace op)
        shorthand (name op)]
    (if (some? ns)
      (str (str/replace ns "." "-") "-" shorthand)
      shorthand)))

(defn assoc-path
  [m op input]
  (let [{:keys [path method] :as cmd} (generate-path input)
        defined-extensions (:extensions input)]
    (assoc-in m [path method]
              (-> cmd
                  (dissoc :path :method)
                  (assoc :operationId (operation-name op))
                  (merge (when defined-extensions
                           (prepare-extensions defined-extensions)))))))

(defn generate-tags
  [tags]
  (->> tags
       (reduce-kv #(assoc %1 %2 (set/rename-keys %3 {:external-docs :externalDocs})) {})
       (reduce-kv #(conj %1 (assoc %3 :name (name %2))) [])
       (map #(prefix-extensions % :extensions))))

(defn- visible? [[_ {:keys [options]}]]
  (get-in options [:openapi?] true))

(defn generate-openapi
  ([api-def]
   (generate-openapi api-def {}))
  ([{:keys      [servers info tags extensions]
     ::reg/keys [resources commands]
    :as        api-def}
    {:keys [filter-commands]}]
   (let [commands (cond->> (filter visible? commands)
                    filter-commands (filter filter-commands))]
     (assoc api-def
            :has-openapi?
            true
            :openapi
            (-> {:openapi    "3.0.0"
                 :info       (prefix-extensions info :extensions)
                 :tags       (generate-tags tags)
                 :components {:schemas (generate-schemas resources)}
                 :servers    servers
                 :extensions extensions
                 :paths      (reduce-kv assoc-path {} (into {} commands))}
                (prefix-extensions :extensions))))))

(defn- encode-key-fn
  [k]
  (let [n (name k)]
    ;;dont apply camelCase, to ensure
    n))

(def object-mapper
  (json/object-mapper {:encode-key-fn encode-key-fn
                       :pretty true}))

(defn json-serialize
  [{:keys [openapi]}]
  (json/write-value-as-string openapi object-mapper))

(defn write-to-file
  [api-def path]
  (spit path (json-serialize api-def)))


(comment

  (require 'blueprint.core)

  (def my-api
    (-> (blueprint.core/parse api-description)
        (generate-openapi)))

  (write-to-file my-api "/home/pyr/foo.json")
  my-api)
