(ns blueprint.handler.reporter
  "
  The default error logger which depends on a configured reporter.
  "
  (:require [spootnik.reporter         :as r]
            [blueprint.handler.reporter.context :as ctx]
            [clojure.tools.logging     :as log]
            [clojure.spec.alpha        :as s]
            [exoscale.ex               :as ex]))

;; This will allow failing early if the config provided to
;; `pluck-fn` is wrong
(s/def ::paths (s/or :map  (s/map-of keyword? coll?)
                     :coll (s/coll-of keyword?)))

(defn pluck-fn
  "A closure which yields a function to select keys out of the context.

   The path specification may either be a collection, in which case, it must
   contain keywords which will be selected out of the context.

   If paths is a map it is expected to a dictionary of key to path in the
   input."
  [paths]
  (ex/assert-spec-valid ::paths paths)
  (if (map? paths)
    (fn [ctx] (reduce-kv #(assoc %1 %2 (get-in ctx %3)) {} paths))
    #(select-keys % paths)))

(defn report-fn
  "Build a reporting function, valid for use as an alternative logger
   for a blueprint error interceptor. Uses a path specification for
   keys to pluck out of the context

   The path specification may either be a collection, in which case, it must
   contain keywords which will be selected out of the context.

   If paths is a map it is expected to a dictionary of key to path in the
   input."
  [paths]
  (let [pluck (pluck-fn paths)]
    (fn [ctx e]
      (try
        (let [plucked (pluck ctx)
              request-id (str (:request-id ctx))
              {::ex/keys [message type]} (ex/datafy e)]
          (ctx/with-context plucked
            (log/errorf e "Error while processing request-id %s, type: %s, message: %s"
                        request-id
                        type
                        message)
            (when-not (:reporter/prevent? (ex-data e))
              (r/capture! {:throwable e
                           :extra plucked}))))
        (catch Exception e
          (log/error e "could not report to sentry"))))))

(def report
  "Default reporting function, only plucks `:request-id` from
   a blueprint context, use `report-fn` to build your reporting
   function if you want more keys to be selected out"
  (report-fn [:request-id]))
