(ns blueprint.handler.auth
  "Common API for the authentication and validation pipelines."
  (:require [exoscale.interceptor      :as ix]
            [exoscale.ex               :as ex]
            [clojure.spec.alpha        :as s]
            [blueprint.handler.default :as default]))

(defprotocol CredentialParser
  :extend-via-metadata true
  "A component to parse the authentication header."
  (parse-credentials! [this auth-header]))

(defprotocol Authenticator
  :extend-via-metadata true
  "A component to validate the creds gotten from the auth header
   and resolve the key & secret out from the ID store."
  (resolve-id! [this creds request])
  (validate-auth! [this id creds request])
  (resolve-keypair! [this id]))

(defprotocol PolicyValidator
  :extend-via-metadata true
  "A component to check if the current API handler
   matches the policy from the id."
  (get-resource-type-ids [this request])
  (validate-policy!
    [this id handler]
    [this id handler resource-type-ids]))

(def path-handler   [:request :handler])
(def path-initiator [:auth/initiator])
(def path-creds     [:auth/creds])
(def path-keypair   [:auth/keypair])
(def path-resources [:auth/resources])
(def path-opt-auth? [:request :blueprint.router/options :auth?])

(defn- auth-required? [ctx]
  (get-in ctx path-opt-auth? true))

(defmacro with-guard
  [f]
  `(ix/when ~f auth-required?))

(defn- get-auth-header!
  [request]
  (or (get-in request [:headers "authorization"])
      (get-in request [:headers :authorization])
      (ex/ex-forbidden! "Auth header not found")))

(defn- get-handler!
  [ctx]
  (or (get-in ctx path-handler)
      (ex/ex-fault! "Cannot find the command in context")))

(def interceptor-parse
  {:name ::parse
   :spec ::ix$parser
   :builder
   (fn [this {:auth/keys [parser]}]
     (cond-> this
       (some? parser)
       (assoc :enter
              (-> (fn [ctx]
                    (let [request (default/original-request ctx)
                          header  (get-auth-header! request)]
                      (parse-credentials! parser header)))
                  (ix/out path-creds)
                  (with-guard)))))})

(def interceptor-auth
  {:name ::auth
   :spec ::ix$authenticator
   :builder
   (fn [this {:auth/keys [authenticator]}]
     (cond-> this
       (some? authenticator)
       (assoc :enter
              (-> (fn [ctx]
                    (let [creds   (get-in ctx path-creds)
                          request (default/original-request ctx)]
                      (ix/execute
                       request
                       [{:leave ::id}
                        {:enter (-> #(resolve-id! authenticator creds %)
                                    (ix/out [::id]))}
                        {:enter (-> #(validate-auth! authenticator (::id %) creds request)
                                    (ix/discard))}])))
                  (ix/out path-initiator)
                  (with-guard)))))})

(def interceptor-policy
  {:name ::policy
   :spec ::ix$policy
   :builder
   (fn [this {:auth/keys [validator]}]
     (cond-> this
       (some? validator)
       (assoc :enter
              (-> (fn [ctx]
                    (let [id      (get-in ctx path-initiator)
                          handler (get-handler! ctx)
                          request (:request ctx)]
                      (ix/execute
                       request
                       [{:leave ::resources}
                        {:enter (-> #(get-resource-type-ids validator %)
                                    (ix/out [::resources]))}
                        {:enter (-> #(validate-policy! validator id handler %)
                                    (ix/lens [::resources]))}])))
                  (ix/out path-resources)
                  (with-guard)))))})

(def interceptor-creds
  {:name ::creds
   :spec ::ix$creds
   :builder
   (fn [this {:auth/keys [authenticator]}]
     (cond-> this
       (some? authenticator)
       (assoc :enter
              (-> (fn [ctx]
                    (let [id (get-in ctx path-initiator)]
                      (resolve-keypair! authenticator id)))
                  (ix/out path-keypair)
                  (with-guard)))))})

(s/def :auth/parser
  (partial satisfies? CredentialParser))

(s/def :auth/authenticator
  (partial satisfies? Authenticator))

(s/def :auth/validator
  (partial satisfies? PolicyValidator))

(s/def ::ix$parser
  (s/keys :opt [:auth/parser]))

(s/def ::ix$authenticator
  (s/keys :opt [:auth/authenticator]))

(s/def ::ix$policy
  (s/keys :opt [:auth/validator]))

(s/def ::ix$creds
  (s/keys :opt [:auth/authenticator]))
