(ns blueprint.handler
  "Facilities to build an asynchronous handler, compatible with
   aleph, out of an interceptor chain.

   This builds an opinionated chain with sane defaults, and a way
   to adapt the chain to the consumer's liking.

   Processing of the chain is done by
   [interceptor](https://github.com/exoscale/interceptor)."
  (:require [blueprint.handler.request-id      :as request-id]
            [blueprint.handler.default         :as default]
            [blueprint.handler.error           :as error]
            [blueprint.handler.format          :as format]
            [blueprint.handler.auth            :as auth]
            [blueprint.core                    :as blueprint]
            [blueprint.router                  :as router]
            [blueprint.openapi                 :as openapi]
            [aleph.http.params                 :as params]
            [exoscale.interceptor              :as ix]
            [exoscale.ex                       :as ex]
            [clojure.spec.alpha                :as s]
            [blueprint.interceptor             :as bpi]))

(def default-interceptors
  "The default interceptor chain. doc/interceptor.md
   should be updated when this changes."
  [error/last-ditch             ;; A last-ditch error catcher
   default/final                ;; Extract :response out
   default/logger
   default/wrap-input-stream    ;; wraps, if enabled, body payload into resettable input stream
   default/add-original-request ;; Add separate field for original request
   request-id/interceptor       ;; Add request ID
   format/enter                 ;; Content negotiation and (de)serialization
   format/leave
   error/interceptor            ;; Catch/log errors
   default/route
   default/service-desc
   auth/interceptor-parse
   auth/interceptor-auth
   default/not-found
   auth/interceptor-policy
   auth/interceptor-creds
   params/interceptor
   default/normalize
   default/transform
   default/wrap-response-body
   default/handler])

(defn build-chain [config]
  (bpi/build-chain default-interceptors config))

(defn build-ring-handler
  "Build a ring handler from a configuration map, optionally
   takes the two required arguments: the API definition and
   handler function as arguments.

   Yields a function of a single request argument which
   processes the chain and yields a final deferred ring
   response"
  [{::keys [chain-fn] :or {chain-fn ix/execute} :as this}]
  (ex/assert-spec-valid ::config this)
  (let [chain (bpi/build-chain default-interceptors this)]
    (assoc this ::chain chain ::ring-handler #(chain-fn {:request %} chain))))

(def ^{:arglists '([config])} ring-handler
  "Fetch the ring handler fn out of a configured server map"
  ::ring-handler)

(def original-request
  "Fetch original request from a context"
  #'default/original-request)

(defprotocol BlueprintDispatcher
  :extend-via-metadata true
  (command-map [this] "Yield a partial dispatch map"))

(defn api-handler-fn
  "The default handler function, at the end of a blueprint
   interceptor chain. When reached, a valid command has been
   parsed.

   A provided dispatch map is looked up for the corresponding
   handler function and hands off to it."
  [definition dispatch-map]
  (let [known-handlers (-> definition :commands keys set)]
    ;; TODO: Rather than returning 501, assert that there's a handler
    ;; for each command
    (fn [{:keys [handler] :as request}]
      (let [handler-fn (get dispatch-map handler)]
        (cond
          (some? handler-fn)
          (handler-fn request)

          (contains? known-handlers handler)
          {:status 501
           :body   {:message (str "command " (str handler) " is missing")}}

          :else
          {:status 404
           :body   {:message "not found"}})))))

(defn find-dispatcher
  "Resolve key to a dispatcher. Prefers protocol implementations,
   namespaced keys, or yields the value"
  [this k]
  (if-some [dispatcher (get this k)]
    (cond
      ;; `::commands` takes precedence as it should be seen as the
      ;; preferred way to provide dependencies
      (some? (::commands dispatcher))             (::commands dispatcher)
      ;; The metadata extensible protocol `BlueprintDispatcher` will be
      ;; deprecated, which will result in the following two conditions to
      ;; be ignored.
      (satisfies? BlueprintDispatcher dispatcher) (command-map dispatcher)
      (contains? (meta dispatcher) `command-map)  (command-map dispatcher)
      ;; Wrapping plain maps in a map of `:blueprint.handler/commands` is
      ;; easy enough that this option will likely be deprecated as well.
      :else                                       dispatcher)
    (ex/ex-not-found! (str "nil dispatcher: " k))))

(defn build-dispatch-map
  "Walk the collection of dispatcher paths, representing keys
   into the input map that should be looked up to find partial
   dispatch maps.

   Elements found may implement BlueprintDispatcher or contain
   a `:blueprint.server/commands` key. When neither of these
   are true, the map is returned as is.

   All provided dispatchers are merged together and should be
   a valid dispatch-map implementation (a map of keyword to
   values implementing `clojure.lang.IFn`."
  [{::keys [dispatch-map dispatchers] :or {dispatch-map {}} :as this}]
  (ex/assert-spec-valid
   ::dispatch-map
   (into dispatch-map (map (partial find-dispatcher this) dispatchers))))

(defn build-handler
  "Constructs a handler function"
  [{:blueprint.core/keys [definition] :as this}]
  (update this
          ::handler
          #(or % (api-handler-fn definition (build-dispatch-map this)))))

(defn start
  "Start a previously configured server (as obtained by `build`)"
  [{::keys [start-fn] :or {start-fn identity} :as this}]
  (-> this
      (update :blueprint.core/definition blueprint/parse)
      (update :blueprint.core/definition router/generate-router)
      (update :blueprint.core/definition openapi/generate-openapi)
      build-handler
      build-ring-handler
      start-fn))

(defn stop
  "Stop a previously configured server"
  [{::keys [stop-fn] :or {stop-fn identity} :as this}]
  (stop-fn this))

(defn build
  "Builds a map describing a stopped API server,
   ready for use by `start` and `stop`.

  Acts as a component if wired to a system map.

  No required checking of configuration yet, this is deferred
  to `start` in case dynamic wiring of dependencies
  is done."
  ([this]
   (with-meta
     this
     {`com.stuartsierra.component/start start
      `com.stuartsierra.component/stop stop}))
  ([start-fn stop-fn]
   (build {} start-fn stop-fn ix/execute))
  ([start-fn stop-fn chain-fn]
   (build {} start-fn stop-fn chain-fn))
  ([m start-fn stop-fn chain-fn]
   (build (assoc m
                 ::start-fn start-fn
                 ::stop-fn stop-fn
                 ::chain-fn chain-fn))))

(s/def ::dispatch-map (s/map-of keyword? ifn?
                                :min-count 1))
(s/def ::dispatchers (s/coll-of keyword?))

(s/def ::disabled (s/coll-of qualified-keyword?))
(s/def ::handler ifn?)
(s/def ::config (s/keys :req [:blueprint.core/definition]
                        :opt [::handler ::bpi/disabled ::bpi/additional ::error/logger-fn]))

(s/def ::build-server (s/keys :opt [:blueprint.core/definition
                                    ::handler
                                    ::dispatch-map
                                    ::dispatchers]))
