(ns blueprint.handler.error
  (:require [exoscale.ex           :as ex]
            [exoscale.interceptor  :as interceptor]
            [clojure.spec.alpha    :as s]
            [clojure.tools.logging :as log]))

;; Utilities
;; =========

(defn- find-message
  "Finds most appropriate message to expose within exception stack."
  [e]
  (loop [e e]
    (let [msg (ex-message e)]
      (cond
        (some? msg) msg
        (some? e)   (recur (ex-cause e))))))

;; Translation of exceptions to responses
;; ======================================

(defmulti ex->format
  #(some-> % ex-data :type)
  :hierarchy ex/hierarchy)

(defmethod ex->format ::ex/incorrect    [_]
  {:http/status 400 :http/message "bad request"  ::reveal? true ::log? false})

(defmethod ex->format ::ex/not-found    [_]
  {:http/status 404 :http/message "not found"    ::reveal? true ::log? false})

(defmethod ex->format ::ex/invalid-spec [_]
  {:http/status 400 :http/message "invalid data" ::reveal? true ::log? false})

(defmethod ex->format ::ex/forbidden    [_]
  {:http/status 403 :http/message "forbidden"    ::reveal? true ::log? false})

(defmethod ex->format ::ex/unsupported  [_]
  {:http/status 405 :http/message "unsupported"  ::reveal? true ::log? false})

(defmethod ex->format ::ex/conflict     [_]
  {:http/status 409 :http/message "conflict"     ::reveal? true ::log? false})

(defmethod ex->format ::ex/unavailable  [_]
  {:http/status 504 :http/message "server error" ::reveal? false ::log? true})

(defmethod ex->format ::ex/busy         [_]
  {:http/status 503 :http/message "busy"         ::reveal? false ::log? true})

(defmethod ex->format :default          [_]
  {:http/status 500 :http/message "server error" ::reveal? false ::log? true})

(defn ex->backward-compat-format
  [e]
  (when (ex/type? ::ex/user-exposable e)
    {::reveal? true}))

(defn reveal-exception
  "Process incoming exception, log if necessary with a provided `logger`
  and produce a valid error message from it."
  [logger ctx e]
  (let [{:http/keys [headers status body message] ::keys [reveal? log?]}
        (merge (ex->format e)
               ;; Still look at ::ex/user-exposable
               (ex->backward-compat-format e)
               (ex-data e))

        payload
        (if reveal? {:message  (find-message e)} (or body {:message message}))]

    (when log?
      (logger ctx e))

    (update ctx
            :response
            (fn [response]
              (cond-> (assoc response
                             :status status
                             :body payload)
                (some? headers)
                (assoc-in [:response :headers] headers))))))

(defn default-logger [ctx e]
  (log/error e "while processing request" (str (:request-id ctx))
             ":" (find-message e)))

(def interceptor
  {:name    ::error
   :spec    ::logger-fn
   :builder (fn [this {::keys [logger]}]
              (assoc this :error (partial reveal-exception (or logger
                                                               default-logger))))})

(defn not-found
  "A helper to build a properly formatted not-found exception"
  ([]
   (ex-info "not found" {:type ::ex/not-found}))
  ([msg]
   (ex-info msg {:type ::ex/not-found})))

(defn last-ditch-response
  [e]
  (let [{:http/keys [status body message headers]}
        (try (ex->format e)
             (catch Exception _ #:http{:status 500 :body "server error"}))]
    (cond-> {:status status :body (or message body)}
      (some? headers)
      (assoc :headers headers))))

(def last-ditch
  {:name    ::last-ditch
   :error   (fn [ctx e]
              (log/error e "early/late error during interceptor chain run"
                         (ex-message e))
              (last-ditch-response e))})

(s/def ::logger fn?)
(s/def ::logger-fn (s/keys :opt [::logger]))
