(ns blueprint.handler.auth
  "Common API for the authentication and validation pipelines."
  (:require [exoscale.interceptor      :as ix]
            [exoscale.ex               :as ex]
            [manifold.deferred         :as d]
            [clojure.spec.alpha        :as s]
            [blueprint.handler.default :as default]))

(defprotocol CredentialParser
  :extend-via-metadata true
  "A component to parse the authentication header."
  (parse-credentials! [this auth-header]))

(defprotocol Authenticator
  :extend-via-metadata true
  "A component to validate the creds gotten from the auth header
   and resolve the key & secret out from the ID store."
  (resolve-id! [this creds request])
  (validate-auth! [this id creds request])
  (resolve-keypair! [this id]))

(defprotocol PolicyValidator
  :extend-via-metadata true
  "A component to check if the current API handler
   matches the policy from the id."
  (get-resource-type-ids [this request])
  (validate-policy!
    [this id handler]
    [this id handler resource-type-ids]))

(def path-handler   [:request :handler])
(def path-initiator [:auth/initiator])
(def path-creds     [:auth/creds])
(def path-keypair   [:auth/keypair])
(def path-resources [:auth/resources])
(def path-opt-auth? [:request :blueprint.router/options :auth?])

(defn- auth-required? [ctx]
  (get-in ctx path-opt-auth? true))

(defmacro with-guard
  [f]
  `(ix/when ~f auth-required?))

(defn- get-auth-header!
  [request]
  (or (get-in request [:headers "authorization"])
      (get-in request [:headers :authorization])
      (ex/ex-forbidden! "Auth header not found")))

(defn- get-handler!
  [ctx]
  (or (get-in ctx path-handler)
      (ex/ex-fault! "Cannot find the command in context")))

(def interceptor-parse
  {:name ::parse
   :spec ::ix$parser
   :builder
   (fn [this {:auth/keys [parser]}]
     (cond-> this
       (some? parser)
       (assoc :enter
              (with-guard
                (fn [ctx]
                  (let [request (default/original-request ctx)
                        header  (get-auth-header! request)]
                    (d/chain
                     (parse-credentials! parser header)
                     (fn [creds]
                       (assoc-in ctx path-creds creds)))))))))})

(def interceptor-auth
  {:name ::auth
   :spec ::ix$authenticator
   :builder
   (fn [this {:auth/keys [authenticator]}]
     (cond-> this
       (some? authenticator)
       (assoc :enter
              (with-guard
                (fn [ctx]
                  (let [creds   (get-in ctx path-creds)
                        request (default/original-request ctx)]
                    (d/chain
                     (resolve-id! authenticator creds request)
                     (fn [id]
                       (d/chain
                        (validate-auth! authenticator id creds request)
                        (constantly id)))
                     (fn [id]
                       (assoc-in ctx path-initiator id)))))))))})

(def interceptor-policy
  {:name ::policy
   :spec ::ix$policy
   :builder
   (fn [this {:auth/keys [validator]}]
     (cond-> this
       (some? validator)
       (assoc :enter
              (with-guard
                (fn [ctx]
                  (let [id      (get-in ctx path-initiator)
                        handler (get-handler! ctx)
                        request (:request ctx)]
                    (d/chain
                     (get-resource-type-ids validator request)
                     (fn [resources]
                       (validate-policy! validator id handler resources))
                     (fn [resolved-resources]
                       (assoc-in ctx path-resources resolved-resources)))))))))})

(def interceptor-creds
  {:name ::creds
   :spec ::ix$creds
   :builder
   (fn [this {:auth/keys [authenticator]}]
     (cond-> this
       (some? authenticator)
       (assoc :enter
              (with-guard
                (fn [ctx]
                  (let [id (get-in ctx path-initiator)]
                    (d/chain
                     (resolve-keypair! authenticator id)
                     (fn [keypair]
                       (assoc-in ctx path-keypair keypair)))))))))})

(s/def :auth/parser
  (partial satisfies? CredentialParser))

(s/def :auth/authenticator
  (partial satisfies? Authenticator))

(s/def :auth/validator
  (partial satisfies? PolicyValidator))

(s/def ::ix$parser
  (s/keys :opt [:auth/parser]))

(s/def ::ix$authenticator
  (s/keys :opt [:auth/authenticator]))

(s/def ::ix$policy
  (s/keys :opt [:auth/validator]))

(s/def ::ix$creds
  (s/keys :opt [:auth/authenticator]))
