(ns blueprint.handler.format
  "Content-type negotiation and parsing, thanks to muuntaja"
  (:require [exoscale.interceptor :as interceptor]
            [exoscale.ex          :as ex]
            [muuntaja.core        :as m]
            [muuntaja.format.core :as mfc]
            [muuntaja.format.json :as mfj]
            [jsonista.core        :as json]
            [exoscale.cloak]
            [clojure.string :as str])
  (:import (com.fasterxml.jackson.databind MapperFeature
                                           SerializationFeature)))

(def safe-json-mapper
  (doto (json/object-mapper
         {:decode-key-fn #(keyword (if-let [idx (str/last-index-of % "/")]
                                     (subs % (inc idx))
                                     %))})
    (.configure SerializationFeature/FAIL_ON_EMPTY_BEANS false)
    (.configure MapperFeature/AUTO_DETECT_GETTERS false)
    (.configure MapperFeature/AUTO_DETECT_IS_GETTERS false)
    (.configure MapperFeature/AUTO_DETECT_SETTERS false)
    (.configure MapperFeature/AUTO_DETECT_FIELDS false)
    (.configure MapperFeature/DEFAULT_VIEW_INCLUSION false)))

(def muuntaja-instance
  (m/create
   (assoc-in m/default-options
             [:formats "application/json"]
             (mfc/map->Format
              {:name "application/json"
               :decoder [mfj/decoder {:mapper safe-json-mapper}]
               :encoder [mfj/encoder {:mapper safe-json-mapper}]}))))

(def known-error-types
  "An exhaustive list of error types thrown by the muntaaja library"
  #{:muuntaja/decode
    :muuntaja/request-charset-negotiation
    :muuntaja/response-charset-negotiation
    :muuntaja/response-format-negotiation})

;; Install derivations for muuntaja, by declaring
;; that the above error types belong to the :exoscale.ex/incorrect
;; category, such errors will result in 400 being thrown
(doseq [error-type known-error-types]
  (ex/derive error-type ::ex/incorrect))

(defn- format-enter
  [{:keys [request] :as context}]
  (let [request (m/negotiate-and-format-request muuntaja-instance request)]
    (assoc context ::request request :request request)))

(defn- format-leave
  [{::keys [request] :keys [response] :as context}]
  (m/format-response muuntaja-instance request response))

(def enter
  {:name  :blueprint.handler/format-enter
   :enter format-enter})

(def leave
  {:name  :blueprint.handler/format-leave
   :leave (interceptor/out format-leave [:response])})
