(ns blueprint.handler
  "Facilities to build an asynchronous handler, compatible with
   aleph, out of an interceptor chain.

   This builds an opinionated chain with sane defaults, and a way
   to adapt the chain to the consumer's liking.

   Processing of the chain is done by
   [interceptor](https://github.com/exoscale/interceptor)."
  (:require [blueprint.handler.request-id      :as request-id]
            [blueprint.handler.default         :as default]
            [blueprint.handler.error           :as error]
            [blueprint.handler.format          :as format]
            [blueprint.handler.auth            :as auth]
            [aleph.http.params                 :as params]
            [exoscale.interceptor.manifold     :as ixm]
            [exoscale.ex                       :as ex]
            [clojure.spec.alpha                :as s]
            [blueprint.interceptor             :as bpi]))

(def default-interceptors
  "The default interceptor chain. doc/interceptor.md
   should be updated when this changes."
  [error/last-ditch             ;; A last-ditch error catcher
   default/final                ;; Extract :response out
   default/logger
   default/add-original-request ;; Add separate field for original request
   request-id/interceptor       ;; Add request ID
   format/enter                 ;; Content negotiation and (de)serialization
   format/leave
   error/interceptor            ;; Catch/log errors
   default/route
   auth/interceptor-parse
   auth/interceptor-auth
   default/not-found
   auth/interceptor-policy
   auth/interceptor-creds
   params/interceptor
   default/normalize
   default/transform
   default/wrap-response-body
   default/handler])

(defn build-chain [config]
  (bpi/build-chain default-interceptors config))

(defn ring-handler
  "Build a ring handler from a configuration map, optionally
   takes the two required arguments: the API definition and
   handler function as arguments.

   Yields a function of a single request argument which
   processes the chain and yields a final deferred ring
   response"
  ([config]
   (ex/assert-spec-valid ::config config)
   (let [chain (bpi/build-chain default-interceptors config)]
     (fn [request]
       (ixm/execute {:request request} chain))))
  ([definition handler]
   (ring-handler definition handler nil))
  ([definition handler config]
   (ring-handler (assoc config :blueprint.core/definition definition ::handler handler))))

(def original-request
  "Fetch original request from a context"
  #'default/original-request)

(s/def ::disabled (s/coll-of qualified-keyword?))
(s/def ::handler ifn?)
(s/def ::config (s/keys :req [:blueprint.core/definition ::handler]
                        :opt [::bpi/disabled ::bpi/additional ::error/logger-fn]))
