(ns blueprint.handler
  (:require [aleph.http                    :as http]
            [aleph.http.params             :as params]
            [blueprint.error               :as error]
            [clojure.tools.logging         :as log]
            [muuntaja.core                 :as m]
            [jsonista.core                 :as json]
            [exoscale.interceptor.manifold :as ixm]
            [exoscale.interceptor          :as interceptor]
            [expound.alpha                 :as expound]
            [spec-tools.core               :as st]
            [manifold.deferred             :as d]
            [clojure.spec.alpha            :as s]
            [exoscale.ex                   :as ex]))

(defn normalize
  [{:keys [get-params body-params path-params handler]}]
  (merge get-params
         path-params
         body-params
         {:handler handler}))

(defn ex-invalid-spec
  [spec input]
  (ex/ex-info (format "Invalid input parameters: %s" (s/explain-str spec input))
              [::invalid-spec [::ex/invalid-spec ::ex/user-exposable]]
              (s/explain-data spec input)))

(defn decode-input
  [spec input]
  (let [decoded (st/decode spec input st/json-transformer)]
    (when (= decoded ::s/invalid)
      (binding [st/*transformer* st/json-transformer
                st/*encode?*     false]
        (throw (ex-invalid-spec spec input))))
    decoded))

(defn interceptor-map
  [api-def handler]
  (let [router (:router api-def)
        specs  (:specs api-def)]
    {::request-id
     {:name  :request-id
      :enter #(assoc % :request-id (java.util.UUID/randomUUID))}

     ::final
     {:name  :final
      :leave :response}

     ::format
     {:name  :format
      :enter (fn [{:keys [request] :as context}]
               (let [request (m/negotiate-and-format-request m/instance request)]
                 (assoc context ::request request :request request)))

      :leave (-> (fn [{::keys [request] :keys [response] :as context}]
                   (m/format-response m/instance request response))
                 (interceptor/out [:response]))}

     ::params
     params/interceptor

     ::route
     {:name  :route
      :enter (-> router
                 (interceptor/lens [:request]))}

     ::normalize
     {:name  :normalize
      :enter (-> normalize
                 (interceptor/lens [:request]))}

     ::transform
     {:name  :transform
      :enter (-> (partial decode-input (:handler specs))
                 (interceptor/lens [:request]))}

     ::handler
     {:name  :handler
      :enter (-> handler
                 (interceptor/in [:request])
                 (interceptor/out [:response]))}

     ::error-response error/exception-response
     ::error-logger error/exception-logger}))

(def default-interceptor-chain [::final
                                ::error-response
                                ::error-logger
                                ::request-id
                                ::format
                                ::params
                                ::route
                                ::normalize
                                ::transform
                                ::handler])

(defn generate-ring-handler
  ([api-def handler] (generate-ring-handler api-def handler {}))
  ([api-def handler {:keys [user-interceptors
                            interceptor-chain
                            default-error-msg]
                      :or {default-error-msg "Internal error"
                           interceptor-chain default-interceptor-chain}}]
   (let [interceptors (merge (interceptor-map api-def handler) user-interceptors)
         chain ((apply juxt interceptor-chain) interceptors)]
     (error/unexposable-handler default-error-msg)
     (fn [request]
       (d/catch (ixm/execute {:request request
                              :default-error-msg default-error-msg}
                             chain)
           (fn [x]
             (log/error x "unhandled http error")
             {:status 500
              :body   default-error-msg}))))))

(comment

  (defn handler
    [request]
    (clojure.pprint/pprint {:request request})
    {:status 200 :body request})

  (require 'blueprint.core)
  (require 'blueprint.router)
  (require 'blueprint.openapi)



  (defn dyn-handler
    []
    (generate-ring-handler
     (-> (blueprint.core/load-schema)
         (blueprint.router/generate-router)
         (blueprint.openapi/generate-oas-map))
     handler))


  (println :foo)
  (def srv
    (http/start-server (dyn-handler) {:port 8080}))

  (.close srv))

