(ns blueprint.registry
  "A data-based registry of specifications for input and commands
   in an HTTP API."
  (:require [clojure.spec.alpha :as s]
            [clojure.edn        :as edn]
            [exoscale.ex        :as ex]
            blueprint.spec))

(def simple-preds
  #{'uuid? 'pos-int? 'string? 'boolean?
    'set? 'map? 'any? 'vector? 'list? 'inst?
    'keyword?})

(defmulti clean-resource (fn [[type v]]
                           (let [res (if (= type :vector) (first v) type)]
                             res)))

(defmethod clean-resource :and
  [[_vector [_and {:keys [defs opts] :as m}]]]
  (cond-> {:type :and
           :defs (vec (for [d defs] (clean-resource d)))}
    (some? opts)
    (assoc :opts opts)))

(defmethod clean-resource :map
  [[_vector [_map {:keys [attributes] :as m}] mapvec]]
  (assoc m :attributes
         (vec (for [attr attributes]
                (update attr :def clean-resource)))))

(defmethod clean-resource :multi
  [[_vector [_multi {:keys [alternatives] :as m}]]]
  (assoc m :alternatives
         (vec (for [alt alternatives]
                (update alt :def clean-resource)))))

(defmethod clean-resource :map-of
  [[_vector [_map-of m]]]
  (-> m
      (update :key-def clean-resource)
      (update :val-def clean-resource)))

(defmethod clean-resource :coll-of
  [[_vector [_coll-of m]]]
  (update m :def clean-resource))

(defmethod clean-resource :or-fields
  [[_vector [_or-fields m]]]
  m)

(defmethod clean-resource :int-condition
  [[_vector [_int-cond m]]]
  m)

(defmethod clean-resource :> [[_ [_ m]]]  m)
(defmethod clean-resource :>= [[_ [_ m]]] m)
(defmethod clean-resource :<= [[_ [_ m]]] m)
(defmethod clean-resource :< [[_ [_ m]]] m)
(defmethod clean-resource :not= [[_ [_ m]]] m)

(defmethod clean-resource :reference
  [[_ reference]]
  {:type :reference :reference reference})

(defmethod clean-resource :ip
  [_]
  {:type :ip})

(defmethod clean-resource :ipv4
  [_]
  {:type :ipv4})

(defmethod clean-resource :ipv6
  [_]
  {:type :ipv6})

(defmethod clean-resource :enum
  [[_ values]]
  {:type :enum :values values})

(defmethod clean-resource :builtin
  [[_ x]]
  {:type :pred :pred (cond
                       (symbol? x)
                       (-> x ns-resolve deref)

                       (vector? x)
                       (second x)

                       :else
                       x)})

(defn clean-entry
  [m k v]
  (assoc m k (clean-resource v)))

(defn clean-command
  [m k {:keys [input params path] :as cmd}]
  (assoc m
         k
         (cond-> (-> cmd
                     (update :output (partial reduce-kv clean-entry {}))
                     (update-in [:path :elems]
                                #(for [[t def] %]
                                   (if (= t :arg)
                                     [:arg {:name (:name def)
                                            :def  (clean-resource (:def def))}]
                                     [t def]))))

           (some? input)
           (update :input clean-resource)
           (some? params)
           (update :params (partial reduce-kv clean-entry {})))))

(defn build
  [source]
  (ex/assert-spec-valid ::blueprint source)
  (let [{:keys [resources commands]} (s/conform ::blueprint source)]
    (assoc source
           ::resources (reduce-kv clean-entry {} resources)
           ::commands  (reduce-kv clean-command {} commands))))
