(ns ^:private blueprint.interceptor
  "Facilities to build an interceptor chain. This namespace is to
  be consumed by other blueprint namespaces."
  (:require [exoscale.ex                       :as ex]
            [blueprint.core                    :as bp]
            [clojure.spec.alpha                :as s]))

(defprotocol BlueprintInterceptorChain
  :extend-via-metadata true
  (interceptor-chain [this] "Yield a partial interceptor chain"))

(defn build-with
  "Interceptor builder, if a `:builder` key is found in the
   interceptor, it will be called on itself, with the configuration
   as a second argument. Validates with spec at `:spec` key if provided.

   This allows providing configuration to the chain"
  [config {:keys [builder spec] :as interceptor}]
  (when spec (ex/assert-spec-valid spec config))
  (cond-> interceptor
    (some? builder) (builder config)))

(defn- insert-at
  "Inserts an element before or after another one in a chain"
  [pos chain target ix]
  (let [[head tail] (split-with #(not= target (:name %)) chain)]
    (cond
      (empty? tail)  (throw (ex/ex-info "no such target" ::ex/incorrect))
      (= pos :after) (concat head [(first tail) ix] (rest tail))
      :else          (concat head [ix] tail))))

(defn add-to-chain
  "Adds an interceptor to the chain, `:blueprint.handler/position` determines
  the position in the chain, and can be:

  - `:first`: Puts the interceptor at the beginning of the chain.
  - `:last`: Puts the interceptor at the end of the chain.
  - `:before`: Puts the interceptor before the one whose name is stored at the
    `:blueprint.interceptor/target` key in the input map.
  - `:after`: Puts the interceptor after the one whose name is stored at the
    `:blueprint.interceptor/target` key in the input map.

  Incorrect parameters will yield to thrown exceptions."
  [chain {::keys [position target] :as interceptor}]
  (vec
   (case position
     :last   (conj chain interceptor)
     :first  (into [interceptor] chain)
     :before (insert-at :before chain target interceptor)
     :after  (insert-at :after chain target interceptor)
     (throw (ex/ex-info "invalid position" ::ex/incorrect)))))

(defn place
  "Helper function to populate an interceptor with position info"
  ([interceptor position]
   (place interceptor position nil))
  ([interceptor position target]
   (cond-> (assoc interceptor ::position position)
     (some? target)
     (assoc ::target target))))

(defn expand-chain
  "Resolve element to an interceptor chain. Works against all
   implementations of BlueprintInterceptorChain, maps containing
   `::interceptor-chain`, vectors are rendered as is, and single
   interceptors are rendered as a vector of themselves."
  [ixdef]
  (cond
    (satisfies? BlueprintInterceptorChain ixdef) (interceptor-chain ixdef)
    (contains? (meta ixdef) `interceptor-chain)  (interceptor-chain ixdef)
    (some? (::interceptor-chain ixdef))          (::interceptor-chain ixdef)
    (sequential? ixdef)                          ixdef
    :else                                        [ixdef]))

(defn build-chain
  "Build an interceptor chain in three phases:

  - Start from the default chain
  - Process additional interceptors provided in `::additional`
  - Removes interceptors by name based on those provided in `::disabled`
  - Call the interceptor build step if any with the provided config"
  [default-interceptors {::keys [additional disabled] :as config}]
  (ex/assert-spec-valid ::config config)
  (->> (mapcat expand-chain additional)
       (ex/assert-spec-valid ::interceptor-coll)
       (reduce add-to-chain default-interceptors)
       (remove #(contains? (set disabled) (:name %)))
       (map (partial build-with config))
       (doall)))

(s/def ::name keyword?)
(s/def ::position #{:first :last :before :after})
(s/def ::target keyword?)
(s/def ::named-interceptor (s/keys :req-un [::name] :opt [::position ::target]))
(s/def ::interceptor-coll (s/coll-of ::named-interceptor))

(s/def ::disabled (s/coll-of qualified-keyword?))

;; a bit overreaching because of the recursive def
;; but tests pass. the `mapcat` call while building the chain
;; makes the chain structure possibly recursive
(s/def ::additional (s/or :proto   #(satisfies? BlueprintInterceptorChain %)
                          :meta    #(contains? (meta %) `interceptor-chain)
                          :chain   #(some? (::interceptor-chain %))
                          :seq     (s/coll-of ::additional)
                          :default ::named-interceptor))

(s/def ::config (s/keys :opt [::disabled ::additional]))
