(ns blueprint.spec-gen
  "A generator of clojure specs for data-based API definitions"
  (:require [clojure.spec.alpha         :as s]
            [spec-tools.core            :as st]
            [spec-tools.data-spec       :as ds])
  (:import (org.apache.commons.validator.routines InetAddressValidator)))

(def ^:dynamic *base-ns* "blueprint.spec-gen.api")
(def ^:dynamic *resources*)

(defn qualify
  [k]
  (if (nil? (namespace k))
    (keyword (name *base-ns*) (name k))
    k))

(def ^:no-doc multi-print-method
  "Used to specify our preferred print-method for a multi ref"
  (delay (get-method print-method clojure.lang.MultiFn)))

(defn ^:no-doc ^clojure.lang.MultiFn multi-ref
  "A variant of clojure.lang.MultiFn that also implements clojure.lang.IRef
   This is used to trick some spec internals which expect proper vars"
  [symname dispatch-fn default-val hierarchy]
  (let [set-print (fn [px f] (.addMethod print-method (class px) f))]
    (doto (proxy [clojure.lang.MultiFn clojure.lang.IRef]
              [symname dispatch-fn default-val hierarchy]
            (deref [] this))
      (set-print @multi-print-method))))

(def genq (comp qualify gensym))
(def genq-name (comp name genq))

(defmulti -compile-spec :type)

(defn export
  [x]
  (ds/spec {:spec (-compile-spec x) :name (genq "dynvar")}))

(def compile-spec #'-compile-spec)

(defmethod -compile-spec :map
  [{:keys [attributes] :as def}]
  (ds/spec
   {:spec
    (reduce (fn [m {:keys [attribute def opts] :as attr}]
              (let [opt? (not (:req? opts))]
                (assoc m
                       (cond-> attribute opt? ds/opt)
                       (compile-spec (:def attr)))))
            {}
            attributes)
    :name (qualify (gensym "anonymous-map"))}))

(defmethod -compile-spec :map-of
  [{:keys [key-def val-def] :as def}]
  (s/map-of (st/create-spec {:spec (compile-spec key-def)
                             :form key-def})
            (st/create-spec {:spec (compile-spec val-def)
                             :form val-def})))


(defmethod -compile-spec :coll-of
  [{:keys [def] :as in}]
  (s/coll-of (st/create-spec {:spec (compile-spec def)
                              :form def})))

(defmethod -compile-spec :enum
  [{:keys [values] :as def}]
  (st/spec {:spec (set values) :type :keyword}))

(defmethod -compile-spec :and
  [{:keys [defs]}]
  (let [[single & tail] defs]
    (cond
      (empty? defs) (s/spec any?)
      (empty? tail) (-compile-spec single)
      :else (s/and-spec-impl defs (mapv export defs) nil))))

(defmethod -compile-spec :reference
  [{:keys [reference]}]
  (let [spec-def (get *resources* reference)]
    (st/create-spec {:spec (-compile-spec spec-def)
                     :form spec-def})))

(defmethod -compile-spec :symbol
  [{:keys [type]}]
  (-> type resolve var-get))

(defmethod -compile-spec :fn
  [{:keys [callable]}]
  callable)

(defmethod -compile-spec :pred
  [{:keys [pred]}]
  (if (symbol? pred)
    (-> pred resolve deref)
    pred))

(defmethod -compile-spec :ip
  [_]
  (st/create-spec {:spec #(-> (InetAddressValidator/getInstance)
                              (.isValid ^String %))
                   :description "IP address"
                   :reason "Invalid IP address"}))

(defmethod -compile-spec :ipv4
  [_]
  (st/create-spec {:spec #(-> (InetAddressValidator/getInstance)
                              (.isValidInet4Address ^String %))
                   :description "IPv4 address"
                   :reason "Invalid IPv4 address"}))

(defmethod -compile-spec :ipv6
  [_]
  (st/create-spec {:spec #(-> (InetAddressValidator/getInstance)
                              (.isValidInet6Address ^String %))
                   :description "IPv6 address"
                   :reason "Invalid IPv6 address"}))

(defmethod -compile-spec :multi
  [{:keys [dispatch alternatives] :as def}]
  (let [hiera   (atom (make-hierarchy))
        symname (genq-name "multi-spec")
        mm      (multi-ref symname dispatch :default hiera)]
    (doseq [{:keys [dispatch-val def]} alternatives
            :let [compiled (export def)]]
      (.addMethod mm dispatch-val (constantly compiled)))
    (s/multi-spec-impl def mm :retag)))

(defmethod -compile-spec :>=   [{:keys [val]}] (s/spec #(>= % val)))
(defmethod -compile-spec :>    [{:keys [val]}] (s/spec #(> % val)))
(defmethod -compile-spec :<=   [{:keys [val]}] (s/spec #(<= % val)))
(defmethod -compile-spec :<    [{:keys [val]}] (s/spec #(< % val)))
(defmethod -compile-spec :not= [{:keys [val]}] (s/spec #(not= % val)))

(defn create-temp-specs
  "A run through all declared resources for the spec
   equivalent of `declare` to avoid circular dependencies"
  [{:keys [info resources]}]
  (doseq [k (keys resources)]
    (s/def-impl (qualify k) 'any? (s/spec any?))))

(defn resource-specs
  [resources]
  (reduce-kv
   #(assoc %1 %2 (ds/spec {:spec (compile-spec %3) :name (qualify %2)}))
   {}
   resources))

(defn command-spec
  [op {:keys [input params path] :as cmd}]
  (let [args       (->> (:elems path)
                        (filter #(= :arg (first %)))
                        (map second))
        remote-map (when (= :reference (:type input))
                     (get *resources* (:reference input)))]
    (compile-spec
     (-> input
         (assoc :type :map)
         (update :attributes
                 concat
                 (:attributes remote-map)
                 (for [{:keys [name def]} args]
                   {:attribute name
                    :def       def
                    :opts      {:req? true}})
                 (for [[k v] params] {:attribute k :def v}))))))

(defn command-specs
  "Build a map of operation name to spec, and add an `handler`
  key for a multi-spec dispatch"
  [commands]
  (let [specs
        (reduce-kv #(assoc %1 %2 (ds/spec {:name (qualify %2)
                                           :spec (command-spec %2 %3)}))
                   {}
                   commands)

        hiera
        (atom (make-hierarchy))

        mm
        (multi-ref (genq-name "multi-ref") :handler :default hiera)]
    ;; Install methods on the multi
    (doseq [[k v] specs]
      (.addMethod ^clojure.lang.MultiFn mm k (constantly v)))
    (.addMethod ^clojure.lang.MultiFn mm :blueprint.core/not-found (constantly any?))

    (assoc specs :handler (ds/spec {:name (genq "handler-spec")
                                    :spec (s/multi-spec-impl
                                           (keys commands)
                                           mm
                                           :handler)}))))

(defn output-spec
  "Build a multispec out of a map of `{status spec-def}`, using a multispec with dispatch
on `status`."
  [command status-map]
  (let [specs
        (reduce-kv (fn [m status def]
                     (let [spec-name  (genq (str command "-" status "-"))
                           spec-def   {:body (compile-spec def)}]
                                 
                       (assoc m status (ds/spec {:name spec-name :spec spec-def}))))
                   {}
                   status-map)

        hiera
        (atom (make-hierarchy))

        mm
        (multi-ref (genq-name "output-multi-ref") :status :default hiera)]

    (doseq [[k v] specs]
      (.addMethod ^clojure.lang.MultiFn mm k (constantly v)))

    (ds/spec {:name (genq "status-spec")
              :spec (s/multi-spec-impl
                     (keys specs)
                     mm
                     :status)})))

(defn output-specs
  "Build a map of operation name to output spec."
  [commands]
  (reduce-kv (fn [m k v] (assoc m k (output-spec k (get v :output))))
             {}
             commands))

(defn generate-specs
  [{:keys [info resources commands] :as api-def}]
  (create-temp-specs api-def)
  (binding [*resources* resources]
    (assoc api-def
           :specs
           (merge
            (resource-specs resources)
            (command-specs  commands))
           :output-specs
           (output-specs commands))))
