(ns blueprint.client.interceptors
  (:require [aleph.http :as http]
            [blueprint.core :as core]
            [clojure.spec.alpha :as s]
            [clojure.tools.logging :as log]
            [exoscale.coax :as cx]
            [exoscale.ex :as ex]
            [exoscale.interceptor :as ix]
            [manifold.deferred :as d]
            [muuntaja.core :as m]))
;;
;; Private
;;
(defn- decodeable?
  "Indicates if the response can be decoded."
  [response]
  (some? (get-in response [:headers "content-type"])))

(defn ctx->command-output-spec
  [{:as _ctx :blueprint.client/keys [command-def]}]
  (let [[ns name] command-def]
    (core/command-output-spec ns name)))

(defn- conform-body
  "Tries to conform the body according to the command's output spec.
  If decoding was unsuccessful, a `:invalid? true` keyval is added to the response, and the original
  body is left as is."
  [spec response]
  (let [payload (cx/coerce spec response)]
    (cond-> payload
      (not (s/valid? spec payload))
      (assoc :invalid? true :explain-data (s/explain-data spec response)))))

(defn- decode-body
  "Decodes the http request body according to content type headers."
  [{:keys [headers] :as response}]
  ;; unfortunately, aleph http returns ring headers (lowercased)
  ;; but muuntaja expects Pascal-Case headers
  ;; so a hack is needed to ensure the m/decode-body-response finds the correct content-type
  ;; the alternative is to call m/decode-response-body with content-type and charset args
  ;; (m/decode response "application/edn" "utf-8"
  (let [norm-headers (assoc-in response [:headers "Content-Type"] (get headers "content-type"))]
    (assoc response
           :body
           (m/decode-response-body norm-headers))))

(defn- try-parse-body
  "Tries to manually parse the body. Does a best-effort to parse the body.
  If the body could not be parsed, returns the raw stream."
  [{:keys [request response] :as ctx}]
  ;; m/decode-response-body, via jackson, barfs if an empty input stream is decoded
  ;; https://github.com/FasterXML/jackson-databind/issues/907
  ;; it's hard to reliably check for an empty stream
  ;; (.available (:body response)) returns zero sometimes (when the stream is not yet :aleph/complete ?)
  ;; so we can't reliably set the body to nil when the response stream says available=0
  (try
    (let [format (or (get-in request [:headers "Accept"])
                     (m/default-format m/instance))
          charset (m/default-charset m/instance)
          body (:body response)
          decoded (m/decode m/instance format body charset)]
      (assoc-in ctx [:response :body] decoded))
    (catch Throwable e
      (log/error e "Could not parse response body")
      ctx)))

(defn- status-code->ex-type
  "Try to resolve a status code to a type. A status code may be mapped to several types,
  so we return the most 'generic'."
  [code]
  (case (long code)
    400 ::ex/incorrect
    404 ::ex/not-found
    403 ::ex/forbidden
    405 ::ex/unsupported
    409 ::ex/conflict
    504 ::ex/unavailable
    503 ::ex/busy
    500 ::ex/default
    ::ex/fault))

(defn- decode-error
  "Tries to parse the HTTP error response, so that the response contains a parsed body instead of a stream."
  [{:keys [request]} e]
  (if-not (:decode-on-error request)
    (d/error-deferred e)
    ;; else
    (try
      (let [data (ex-data e)
            status (:status data)
            msg (ex-message e)
            decoded (m/decode-response-body data)
            ;; this has a problem, if the content-type is json
            ;; some *values* won't be reconverted as keywords
            ;; so we use coax to try and coerce the data
            coerced (cx/coerce ::ex/ex-map decoded)]
        (d/error-deferred
         (if (s/valid? ::ex/ex-map coerced)
           ;; no need to assoc :body because m/decode-response-body will consume the input stream
           ;; it would appear as an empty body
           (ex/map->ex-info (assoc-in coerced [:exoscale.ex/data :status] status) {:derive? true})
           (ex/ex-info msg
                       (status-code->ex-type status)
                       (assoc decoded :status status)))))

      (catch Exception ex
        (log/error ex "Could not parse error response body")
        ;; return original exception
        (d/error-deferred e)))))

(defn- http-call [{:keys [request] :as ctx}]
  (-> (d/chain (http/request request)
               #(assoc ctx :response %))
      (d/catch #(decode-error ctx %))))

(def execute-request
  {:name ::execute-request
   :enter #'http-call})

(def lens-response
  {:name ::lens-response
   :enter (ix/lens identity [:response])})

(def get-response
  {:name ::get-response
   :leave (ix/in identity [:response])})

(def parse
  {:name ::parse-body
   :enter (-> try-parse-body
              (ix/when #(not (decodeable? (:response %)))))})

(def decode
  {:name ::decode
   :enter (-> (fn [ctx]
                (assoc ctx
                       :response
                       (cx/coerce (ctx->command-output-spec ctx)
                                  (-> ctx :response decode-body))))
              (ix/when #(decodeable? (:response %))))})

(def decode+conform
  {:name ::decode+conform
   :enter (-> (fn [ctx]
                (assoc ctx
                       :response
                       (conform-body (ctx->command-output-spec ctx)
                                     (-> ctx :response decode-body))))
              (ix/when #(decodeable? (:response %))))})
