(ns blueprint.client
  (:require [blueprint.client.url :as url]
            [blueprint.client.validation :as validation]
            [blueprint.core :as core]
            [blueprint.registry :as reg]
            [aleph.http :as http]
            [muuntaja.core :as m]
            [exoscale.interceptor :as ix]
            [exoscale.interceptor.manifold :as ixm]
            [blueprint.client.spec-gen :as bsg]
            [spec-tools.core :as st]
            [clojure.spec.alpha :as s]
            [clojure.tools.logging :as log]))
;;
;; Private
;;
(defn- decodeable?
  "Indicates if the response can be decoded."
  [response]
  (some? (get-in response [:headers "content-type"])))

(defn- conform-body
  "Tries to conform the body according to the command's output spec.
  If decoding was unsuccessful, a `:invalid? true` keyval is added to the response, and the original
  body is left as is."
  [multispec response]
  (let [decoded  (st/decode multispec response st/json-transformer)
        invalid? (s/invalid? decoded)]
    (if invalid? (assoc response :invalid? true)
        decoded)))

(defn- decode-body
  "Decodes the http request body according to content type headers."
  [{:keys [headers] :as response}]
  ;; unfortunately, aleph http returns ring headers (lowercased)
  ;; but muuntaja expects Pascal-Case headers
  ;; so a hack is needed to ensure the m/decode-body-response finds the correct content-type
  ;; the alternative is to call m/decode-response-body with content-type and charset args
  ;; (m/decode response "application/edn" "utf-8"
  (let [norm-headers (assoc-in response [:headers "Content-Type"] (get headers "content-type"))]
    (->> norm-headers
         (m/decode-response-body)
         (assoc response :body))))

(defn- try-parse-body
  "Tries to manually parse the body. Does a best-effort to parse the body.
  If the body could not be parsed, returns the raw stream."
  [{:keys [request response] :as ctx}]
  ;; m/decode-response-body, via jackson, barfs if an empty input stream is decoded
  ;; https://github.com/FasterXML/jackson-databind/issues/907
  ;; it's hard to reliably check for an empty stream
  ;; (.available (:body response)) returns zero sometimes (when the stream is not yet :aleph/complete ?)
  ;; so we can't reliably set the body to nil when the response stream says available=0
  (try
    (let [format  (or (get-in request [:headers "Accept"])
                    (m/default-format m/instance))
          charset (m/default-charset m/instance)
          body    (:body response)
          decoded (m/decode m/instance format body charset)]
      (assoc-in ctx [:response :body] decoded))
    (catch Throwable e
      (log/error e "Could not parse response body")
      ctx)))

(defn- interceptor-map
  "Builds the interceptor map"
  [response-multispec client]
  (let [conform-body (partial conform-body response-multispec)]
    {::execute-request
     {:name :execute-request
      :enter (-> http/request
                 (ix/in  [:request])
                 (ix/out [:response]))}

     ::lens-response
     {:name  :lens-response
      :enter (ix/lens identity [:response])}

     ::get-response
     {:name  :get-response
      :leave (ix/in identity [:response])}

     ::decode-body
     {:name  :decode-body
      :enter (-> (comp conform-body decode-body)
                 (ix/lens [:response])
                 (ix/when #(decodeable? (:response %))))}

     ::parse-body
     {:name :parse-body
      :enter (-> try-parse-body
                 (ix/when #(not (decodeable? (:response %)))))}}))


(def default-interceptor-chain
  [::execute-request
   ::lens-response
   ::decode-body
   ::parse-body
   ::get-response])

;;
;; Http Client
;;

(defn- base-url
  "Gets the base url for the defined api."
  [{:keys [servers] :as parsed-api}]
  (if-let [prefix (get-in parsed-api [:blueprint.options :path-prefix])]
    (str (:url (first servers)) prefix)
    (:url (first servers))))

(defrecord Client [parsed-api command-defs servers connection-pool])

(defn make-client
  "Create a blueprint client from a blueprint api definition."
  ([api-def] (make-client api-def {}))
  ([api-def {:keys [ssl-context] :as default-options}]
   (let [parsed-api   (core/parse api-def)
         command-defs (bsg/apidef->commands-map parsed-api)
         connection-pool (if ssl-context
                           (http/connection-pool
                            {:connection-options {:ssl-context ssl-context}})
                           http/default-connection-pool)]
     (->Client parsed-api command-defs (:servers api-def) connection-pool))))

(def default-options
  {:pool-timeout       10e3
   :connection-timeout 10e3
   :request-timeout    10e3
   :read-timeout       10e3})

(defn invoke
  "Invoke a `command` on the remote blueprint-based api.
  Clients are created with `(blueprint.client/make-client api-def)`.
  Supported options in `opts`:

  | key                   | description |
  | ----------------------|-------------|
  | `:input`              | The request input to be sent
  | `:server`             | The remote server endpoint (eg: \"http://localhost:8080\"); default is first entry from api definition
  | `:headers`            | A map of additional headers to be sent
  | `:options`            | A map of additional options to add to the request
  | `:as`                 | The format for content negotation; accepts `edn` or `json`; default: `json`
  | `:user-interceptors`  | A map of custom user interceptors to apply on the response
  | `:interceptor-chain`  | The interceptor chain to apply; default: `default-interceptor-chain`
  | `:throw-exceptions`   | Control throwing of exceptions in case of errors
  "
  [{:keys [parsed-api command-defs] :as client}
   command
   {:keys [input
           server
           headers
           options
           as
           throw-exceptions
           user-interceptors
           interceptor-chain]
    :or   {input   nil
           server  (base-url parsed-api)
           headers {}
           as      :json
           throw-exceptions true
           interceptor-chain default-interceptor-chain}
    :as   opts}]

  ;;ensure command is valid
  (validation/validate-command (into #{} (keys command-defs)) command)

  (let [command-def              (get command-defs command)
        {::reg/keys [commands]} parsed-api
        {:keys [input-spec
                path-spec
                params-spec
                response-spec
                input?
                params?
                pathelems?]}     command-def
        command-map              (get commands command)
        request-map              (url/cmd->request command-map input)
        format                   (cond
                                   (= as :edn) "application/edn"
                                   :else       "application/json")]

    (if input? (validation/validate-input input-spec (:body request-map)))
    (if pathelems? (validation/validate-path path-spec input))
    (if params? (validation/validate-params params-spec (:query-params request-map)))

    (let [target-url (str server (:url request-map))
          headers    (cond-> (assoc headers "Accept" format)
                       input? (assoc "Content-Type" format))
          base-req   (-> request-map
                         (merge default-options options)
                         (assoc :url target-url
                                :headers headers
                                :throw-exceptions throw-exceptions
                                :pool (:connection-pool client)))
          http-req   (cond-> base-req
                       input?  (assoc :body (m/encode format input))
                       params? (assoc :query-params (:query-params request-map)))

          ;; build the interceptors chain
          command-interceptors (merge (interceptor-map response-spec client) user-interceptors)
          interceptors         ((apply juxt interceptor-chain) command-interceptors)]

      (ixm/execute {:request http-req
                    #_#_:response (http/request http-req)}
        interceptors))))

(comment
  "end")

(comment
  (ixm/execute {:k 1}
               [{:enter (fn [c] (println "1enter") c)
                 :leave (fn [c] (println "1leave") c)}
                {:enter (fn [c] (println "2enter") c)
                 :leave (fn [c] (println "2leave") c)}]))
